<?php
session_start();
require_once '../database/config.php';

// Restrict access to Admin, DoF, CMD
$role = $_SESSION['role'] ?? '';
if (!in_array($role, ['Admin', 'Finance', 'CMD'])) {
    die('Access denied.');
}

// Get available months
$months = $pdo->query("SELECT DISTINCT DATE_FORMAT(payroll_month, '%Y-%m') as month FROM payrolls ORDER BY month DESC")->fetchAll(PDO::FETCH_COLUMN);
$m1 = $_GET['month1'] ?? ($months[0] ?? '');
$m2 = $_GET['month2'] ?? '';

// Utility: get payroll summary for a month
function getPayrollMatrix($pdo, $month) {
    $payroll_ids = $pdo->prepare("SELECT id FROM payrolls WHERE DATE_FORMAT(payroll_month, '%Y-%m') = ?");
    $payroll_ids->execute([$month]);
    $ids = $payroll_ids->fetchAll(PDO::FETCH_COLUMN);
    if (!$ids) return [];

    $in = implode(',', array_fill(0, count($ids), '?'));

    // Employees paid
    $total_emp = $pdo->prepare("SELECT COUNT(DISTINCT employee_id) FROM payroll_details WHERE payroll_id IN ($in)");
    $total_emp->execute($ids);
    $total_employees = $total_emp->fetchColumn();

    // Main aggregates
    $agg = $pdo->prepare("SELECT 
        SUM(base_salary + total_allowance) as gross_pay,
        SUM(total_allowance) as total_allowance,
        SUM(total_deduction) as total_deduction
    FROM payroll_details WHERE payroll_id IN ($in)");
    $agg->execute($ids);
    $row = $agg->fetch(PDO::FETCH_ASSOC);

    // Breakdown by allowance/deduction type
    $breakStmt = $pdo->prepare("SELECT allowance_breakdown, deduction_breakdown, employee_id FROM payroll_details WHERE payroll_id IN ($in)");
    $breakStmt->execute($ids);
    $allow_type = []; $deduct_type = [];
    $emp_ids = [];
    while ($b = $breakStmt->fetch(PDO::FETCH_ASSOC)) {
        $a = json_decode($b['allowance_breakdown'] ?? '{}', true) ?: [];
        $d = json_decode($b['deduction_breakdown'] ?? '{}', true) ?: [];
        foreach ($a as $k => $v) $allow_type[$k] = ($allow_type[$k] ?? 0) + $v;
        foreach ($d as $k => $v) $deduct_type[$k] = ($deduct_type[$k] ?? 0) + $v;
        $emp_ids[] = $b['employee_id'];
    }

    // Analysis: scheme-wise, level, step, contract staff
    $analysis = [];
    if ($emp_ids) {
        $emp_in = implode(',', array_fill(0, count($emp_ids), '?'));
        $sql = "SELECT scheme, grade_level, step, employment_type, COUNT(*) as count
                FROM employees
                WHERE id IN ($emp_in)
                GROUP BY scheme, grade_level, step, employment_type";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($emp_ids);
        while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $sch = $r['scheme'] ?: 'N/A';
            $level = $r['grade_level'] ?: 'N/A';
            $step = $r['step'] ?: 'N/A';
            $type = $r['employment_type'] ?: 'N/A';
            $analysis[$sch][$level][$step][$type] = $r['count'];
        }
    }

    return [
        'total_employees' => $total_employees ?: 0,
        'gross_pay' => $row['gross_pay'] ?? 0,
        'total_allowance' => $row['total_allowance'] ?? 0,
        'total_deduction' => $row['total_deduction'] ?? 0,
        'allow_types' => $allow_type,
        'deduct_types' => $deduct_type,
        'analysis' => $analysis,
    ];
}

$m1_data = $m1 ? getPayrollMatrix($pdo, $m1) : [];
$m2_data = $m2 ? getPayrollMatrix($pdo, $m2) : [];

// Helper for diff coloring
function diffCell($v1, $v2) {
    $diff = $v1 - $v2;
    if ($diff == 0) return '<span>'.$diff.'</span>';
    $color = $diff > 0 ? 'green' : 'red';
    $sign = $diff > 0 ? '+' : '';
    return '<span style="color:'.$color.'">'.$sign.number_format($diff).'</span>';
}

// Helper to display analysis
function displayAnalysis($analysis) {
    if (!$analysis) return '<div class="alert alert-warning">No analysis data.</div>';
    $out = '';
    foreach ($analysis as $scheme => $levels) {
        $out .= "<div class='card my-3'><div class='card-header'><strong>Scheme: $scheme</strong></div><div class='card-body'>";
        foreach ($levels as $level => $steps) {
            $out .= "<h6>Grade Level: $level</h6><table class='table table-sm table-bordered'><thead><tr><th>Step</th><th>Employment Type</th><th>Count</th></tr></thead><tbody>";
            foreach ($steps as $step => $types) {
                foreach ($types as $type => $count) {
                    $rowstyle = ($type === 'Contract') ? ' style="background:#fcf8e3;"' : '';
                    $out .= "<tr$rowstyle><td>$step</td><td>$type</td><td>$count</td></tr>";
                }
            }
            $out .= "</tbody></table>";
        }
        $out .= "</div></div>";
    }
    return $out;
}

include 'header.php';
?>
<div class="container py-4">
    <h4>Payroll Matrix</h4>
    <form method="get" class="row g-2 mb-4">
        <div class="col-md-4">
            <label>Month 1</label>
            <select name="month1" class="form-select" required>
                <option value="">Select month</option>
                <?php foreach ($months as $m): ?>
                    <option value="<?= $m ?>" <?= $m1==$m?'selected':'' ?>><?= date('F Y', strtotime($m.'-01')) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-4">
            <label>Month 2 (Optional)</label>
            <select name="month2" class="form-select">
                <option value="">-- None (single month analysis) --</option>
                <?php foreach ($months as $m): ?>
                    <option value="<?= $m ?>" <?= $m2==$m?'selected':'' ?>><?= date('F Y', strtotime($m.'-01')) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2 align-self-end">
            <button class="btn btn-primary">Analyze</button>
        </div>
    </form>

    <?php if ($m1 && !$m2 && $m1_data): ?>
        <!-- Single Month Analysis -->
        <div class="card mb-4">
            <div class="card-body table-responsive">
                <div class="mb-4">
                    <button class="btn btn-outline-secondary" onclick="window.print()">
                        <i class="bi bi-printer"></i> Print Report
                    </button>
                </div>
                <div class="row mb-4">
                    <div class="col-md-6 mb-3">
                        <canvas id="singleBarChart"></canvas>
                    </div>
                    <div class="col-md-3 mb-3">
                        <h6 class="text-center">Allowances Breakdown (<?= date('F Y', strtotime($m1.'-01')) ?>)</h6>
                        <canvas id="allowPie1"></canvas>
                    </div>
                    <div class="col-md-3 mb-3">
                        <h6 class="text-center">Deductions Breakdown (<?= date('F Y', strtotime($m1.'-01')) ?>)</h6>
                        <canvas id="deductPie1"></canvas>
                    </div>
                </div>
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th></th>
                            <th><?= date('F Y', strtotime($m1.'-01')) ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><strong>Total Employees Paid</strong></td>
                            <td><?= $m1_data['total_employees'] ?></td>
                        </tr>
                        <tr>
                            <td><strong>Total Gross Pay</strong></td>
                            <td>₦<?= number_format($m1_data['gross_pay'],2) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Total Allowances</strong></td>
                            <td>₦<?= number_format($m1_data['total_allowance'],2) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Total Deductions</strong></td>
                            <td>₦<?= number_format($m1_data['total_deduction'],2) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Deductions by Type</strong></td>
                            <td>
                                <?php foreach ($m1_data['deduct_types'] as $k=>$v): ?>
                                    <?= htmlspecialchars($k) ?>: ₦<?= number_format($v,2) ?><br>
                                <?php endforeach; ?>
                            </td>
                        </tr>
                        <tr>
                            <td><strong>Allowances by Type</strong></td>
                            <td>
                                <?php foreach ($m1_data['allow_types'] as $k=>$v): ?>
                                    <?= htmlspecialchars($k) ?>: ₦<?= number_format($v,2) ?><br>
                                <?php endforeach; ?>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <h5>Analysis for <?= date('F Y', strtotime($m1.'-01')) ?></h5>
                <?= displayAnalysis($m1_data['analysis']) ?>
            </div>
        </div>
    <?php elseif ($m1 && $m2 && $m1_data && $m2_data): ?>
        <!-- Two Month Comparison -->
        <div class="card mb-4">
            <div class="card-body table-responsive">
                <div class="mb-4">
                    <button class="btn btn-outline-secondary" onclick="window.print()">
                        <i class="bi bi-printer"></i> Print Report
                    </button>
                </div>
                <div class="row mb-4">
                    <div class="col-md-6 mb-3">
                        <canvas id="barChart"></canvas>
                    </div>
                    <div class="col-md-3 mb-3">
                        <h6 class="text-center">Allowances Breakdown (<?= date('F Y', strtotime($m1.'-01')) ?>)</h6>
                        <canvas id="allowPie1"></canvas>
                    </div>
                    <div class="col-md-3 mb-3">
                        <h6 class="text-center">Allowances Breakdown (<?= date('F Y', strtotime($m2.'-01')) ?>)</h6>
                        <canvas id="allowPie2"></canvas>
                    </div>
                </div>
                <div class="row mb-4">
                    <div class="col-md-3 mb-3">
                        <h6 class="text-center">Deductions Breakdown (<?= date('F Y', strtotime($m1.'-01')) ?>)</h6>
                        <canvas id="deductPie1"></canvas>
                    </div>
                    <div class="col-md-3 mb-3">
                        <h6 class="text-center">Deductions Breakdown (<?= date('F Y', strtotime($m2.'-01')) ?>)</h6>
                        <canvas id="deductPie2"></canvas>
                    </div>
                </div>
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th></th>
                            <th><?= date('F Y', strtotime($m1.'-01')) ?></th>
                            <th><?= date('F Y', strtotime($m2.'-01')) ?></th>
                            <th>Difference</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><strong>Total Employees Paid</strong></td>
                            <td><?= $m1_data['total_employees'] ?></td>
                            <td><?= $m2_data['total_employees'] ?></td>
                            <td><?= diffCell($m1_data['total_employees'], $m2_data['total_employees']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Total Gross Pay</strong></td>
                            <td>₦<?= number_format($m1_data['gross_pay'],2) ?></td>
                            <td>₦<?= number_format($m2_data['gross_pay'],2) ?></td>
                            <td><?= diffCell($m1_data['gross_pay'], $m2_data['gross_pay']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Total Allowances</strong></td>
                            <td>₦<?= number_format($m1_data['total_allowance'],2) ?></td>
                            <td>₦<?= number_format($m2_data['total_allowance'],2) ?></td>
                            <td><?= diffCell($m1_data['total_allowance'], $m2_data['total_allowance']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Total Deductions</strong></td>
                            <td>₦<?= number_format($m1_data['total_deduction'],2) ?></td>
                            <td>₦<?= number_format($m2_data['total_deduction'],2) ?></td>
                            <td><?= diffCell($m1_data['total_deduction'], $m2_data['total_deduction']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Deductions by Type</strong></td>
                            <td>
                                <?php foreach ($m1_data['deduct_types'] as $k=>$v): ?>
                                    <?= htmlspecialchars($k) ?>: ₦<?= number_format($v,2) ?><br>
                                <?php endforeach; ?>
                            </td>
                            <td>
                                <?php foreach ($m2_data['deduct_types'] as $k=>$v): ?>
                                    <?= htmlspecialchars($k) ?>: ₦<?= number_format($v,2) ?><br>
                                <?php endforeach; ?>
                            </td>
                            <td>
                                <?php
                                $types = array_unique(array_merge(array_keys($m1_data['deduct_types']), array_keys($m2_data['deduct_types'])));
                                foreach ($types as $k) {
                                    $v1 = $m1_data['deduct_types'][$k] ?? 0;
                                    $v2 = $m2_data['deduct_types'][$k] ?? 0;
                                    echo htmlspecialchars($k) . ': ' . diffCell($v1, $v2) . '<br>';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <td><strong>Allowances by Type</strong></td>
                            <td>
                                <?php foreach ($m1_data['allow_types'] as $k=>$v): ?>
                                    <?= htmlspecialchars($k) ?>: ₦<?= number_format($v,2) ?><br>
                                <?php endforeach; ?>
                            </td>
                            <td>
                                <?php foreach ($m2_data['allow_types'] as $k=>$v): ?>
                                    <?= htmlspecialchars($k) ?>: ₦<?= number_format($v,2) ?><br>
                                <?php endforeach; ?>
                            </td>
                            <td>
                                <?php
                                $types = array_unique(array_merge(array_keys($m1_data['allow_types']), array_keys($m2_data['allow_types'])));
                                foreach ($types as $k) {
                                    $v1 = $m1_data['allow_types'][$k] ?? 0;
                                    $v2 = $m2_data['allow_types'][$k] ?? 0;
                                    echo htmlspecialchars($k) . ': ' . diffCell($v1, $v2) . '<br>';
                                }
                                ?>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <div class="row">
                    <div class="col-md-6">
                        <h5><?= date('F Y', strtotime($m1.'-01')) ?> Analysis</h5>
                        <?= displayAnalysis($m1_data['analysis']) ?>
                    </div>
                    <div class="col-md-6">
                        <h5><?= date('F Y', strtotime($m2.'-01')) ?> Analysis</h5>
                        <?= displayAnalysis($m2_data['analysis']) ?>
                    </div>
                </div>
            </div>
        </div>
    <?php elseif ($m1 && $m2): ?>
        <div class="alert alert-warning">No payroll data found for one or both selected months.</div>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    <?php if ($m1 && !$m2 && $m1_data): ?>
    // Single month bar chart
    var ctx = document.getElementById('singleBarChart').getContext('2d');
    new Chart(ctx, {
        type: 'bar',
        data: {
            labels: ['Employees Paid', 'Gross Pay', 'Allowances', 'Deductions'],
            datasets: [{
                label: '<?= date('F Y', strtotime($m1.'-01')) ?>',
                data: [
                    <?= $m1_data['total_employees'] ?>,
                    <?= $m1_data['gross_pay'] ?>,
                    <?= $m1_data['total_allowance'] ?>,
                    <?= $m1_data['total_deduction'] ?>
                ],
                backgroundColor: 'rgba(52, 152, 219, 0.7)'
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: { display: false },
                title: { display: true, text: 'Payroll Analysis' }
            }
        }
    });

    // Pie Charts for Allowance & Deduction Breakdown
    var allowTypes1 = <?= json_encode($m1_data['allow_types']) ?>;
    new Chart(document.getElementById('allowPie1').getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(allowTypes1),
            datasets: [{
                data: Object.values(allowTypes1),
                backgroundColor: ['#1abc9c','#2ecc71','#3498db','#e67e22','#e74c3c','#9b59b6','#34495e','#f39c12','#7f8c8d'],
            }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
    });

    var deductTypes1 = <?= json_encode($m1_data['deduct_types']) ?>;
    new Chart(document.getElementById('deductPie1').getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(deductTypes1),
            datasets: [{
                data: Object.values(deductTypes1),
                backgroundColor: ['#16a085','#27ae60','#2980b9','#d35400','#c0392b','#8e44ad','#2c3e50','#f1c40f','#95a5a6'],
            }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
    });
    <?php elseif ($m1 && $m2 && $m1_data && $m2_data): ?>
    // Two month charts
    var barCtx = document.getElementById('barChart').getContext('2d');
    var barChart = new Chart(barCtx, {
        type: 'bar',
        data: {
            labels: ['Employees Paid', 'Gross Pay', 'Allowances', 'Deductions'],
            datasets: [
                {
                    label: '<?= date('F Y', strtotime($m1.'-01')) ?>',
                    data: [
                        <?= $m1_data['total_employees'] ?>,
                        <?= $m1_data['gross_pay'] ?>,
                        <?= $m1_data['total_allowance'] ?>,
                        <?= $m1_data['total_deduction'] ?>
                    ],
                    backgroundColor: 'rgba(52, 152, 219, 0.7)'
                },
                {
                    label: '<?= date('F Y', strtotime($m2.'-01')) ?>',
                    data: [
                        <?= $m2_data['total_employees'] ?>,
                        <?= $m2_data['gross_pay'] ?>,
                        <?= $m2_data['total_allowance'] ?>,
                        <?= $m2_data['total_deduction'] ?>
                    ],
                    backgroundColor: 'rgba(241, 196, 15, 0.7)'
                }
            ]
        },
        options: {
            responsive: true,
            plugins: {
                legend: { position: 'top' },
                title: { display: true, text: 'Payroll Comparison' }
            }
        }
    });

    // Pie Charts for Allowance Breakdown
    var allowTypes1 = <?= json_encode($m1_data['allow_types']) ?>;
    var allowTypes2 = <?= json_encode($m2_data['allow_types']) ?>;
    new Chart(document.getElementById('allowPie1').getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(allowTypes1),
            datasets: [{
                data: Object.values(allowTypes1),
                backgroundColor: ['#1abc9c','#2ecc71','#3498db','#e67e22','#e74c3c','#9b59b6','#34495e','#f39c12','#7f8c8d'],
            }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
    });
    new Chart(document.getElementById('allowPie2').getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(allowTypes2),
            datasets: [{
                data: Object.values(allowTypes2),
                backgroundColor: ['#1abc9c','#2ecc71','#3498db','#e67e22','#e74c3c','#9b59b6','#34495e','#f39c12','#7f8c8d'],
            }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
    });

    // Pie Charts for Deduction Breakdown
    var deductTypes1 = <?= json_encode($m1_data['deduct_types']) ?>;
    var deductTypes2 = <?= json_encode($m2_data['deduct_types']) ?>;
    new Chart(document.getElementById('deductPie1').getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(deductTypes1),
            datasets: [{
                data: Object.values(deductTypes1),
                backgroundColor: ['#16a085','#27ae60','#2980b9','#d35400','#c0392b','#8e44ad','#2c3e50','#f1c40f','#95a5a6'],
            }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
    });
    new Chart(document.getElementById('deductPie2').getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: Object.keys(deductTypes2),
            datasets: [{
                data: Object.values(deductTypes2),
                backgroundColor: ['#16a085','#27ae60','#2980b9','#d35400','#c0392b','#8e44ad','#2c3e50','#f1c40f','#95a5a6'],
            }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
    });
    <?php endif; ?>
});
</script>
<?php include 'footer.php'; ?>
