<?php
require_once '../database/config.php';

$user_id = $_SESSION['user_id'] ?? null;

// Ensure schema supports c_level and c_step
function ensureCGradeColumns($pdo) {
    try {
        $dbName = $pdo->query("SELECT DATABASE()")->fetchColumn();
        $q = $pdo->prepare("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'employees' AND COLUMN_NAME IN ('c_level','c_step')");
        $q->execute([$dbName]);
        $existing = array_map(function($r){ return $r['COLUMN_NAME']; }, $q->fetchAll(PDO::FETCH_ASSOC));
        $hasLevel = in_array('c_level', $existing, true);
        $hasStep  = in_array('c_step', $existing, true);
        if (!$hasLevel) {
            $pdo->exec("ALTER TABLE employees ADD COLUMN c_level INT NULL");
        }
        if (!$hasStep) {
            $pdo->exec("ALTER TABLE employees ADD COLUMN c_step INT NULL");
        }
    } catch (Exception $e) {
        // Swallow errors to avoid breaking page; log if logger exists
    }
}

ensureCGradeColumns($pdo);

// Function to automatically increase step of all eligible employees
function incrementEmployeeSteps($pdo, $user_id) {
    try {
        // Get current date
        $currentDate = date('Y-m-d');
        
        // Find employees eligible for step increment
        // Conditions: step < 15 AND (last_step_increased_at is empty OR >= 6 months ago)
        $sql = "SELECT id, employee_no, full_name, step, last_step_increased_at 
                FROM employees 
                WHERE step < 15 
                AND (last_step_increased_at = '' 
                     OR last_step_increased_at IS NULL 
                     OR DATE_ADD(STR_TO_DATE(last_step_increased_at, '%Y-%m-%d'), INTERVAL 6 MONTH) <= ?)
                AND status = 'Active'";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentDate]);
        $eligibleEmployees = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $updatedCount = 0;
        
        foreach ($eligibleEmployees as $employee) {
            // Increment step by 1
            $newStep = $employee['step'] + 1;
            
            // Update employee record
            $updateSql = "UPDATE employees 
                         SET step = ?, last_step_increased_at = ? 
                         WHERE id = ?";
            
            $updateStmt = $pdo->prepare($updateSql);
            if ($updateStmt->execute([$newStep, $currentDate, $employee['id']])) {
                $updatedCount++;
                // Log the action
                log_audit($pdo, $user_id, "Auto-incremented step for employee {$employee['employee_no']} ({$employee['full_name']}) from {$employee['step']} to {$newStep}");
            }
        }
        
        return [
            'success' => true,
            'message' => "Successfully incremented steps for {$updatedCount} employees.",
            'updated_count' => $updatedCount
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Error incrementing employee steps: ' . $e->getMessage(),
            'updated_count' => 0
        ];
    }
}

// Handle step increment request
if (isset($_POST['increment_steps'])) {
    $result = incrementEmployeeSteps($pdo, $user_id);
    
    // Store result in session to display message
    $_SESSION['step_increment_result'] = $result;
    
    // Redirect to employee list
    echo "<script>window.location.href='employee_list.php';</script>";
    echo "<noscript><meta http-equiv='refresh' content='0;url=employee_list.php'></noscript>";
    exit;
}

// Fetch filter options
$departments = $pdo->query("SELECT DISTINCT department FROM employees WHERE department IS NOT NULL AND department != '' ORDER BY department")->fetchAll(PDO::FETCH_COLUMN);
$grades = $pdo->query("SELECT DISTINCT grade_level FROM employees WHERE grade_level IS NOT NULL AND grade_level != '' ORDER BY grade_level")->fetchAll(PDO::FETCH_COLUMN);
// For modal selects
$schemes = $pdo->query("SELECT DISTINCT scheme FROM salary_structures WHERE scheme IS NOT NULL AND scheme <> '' ORDER BY scheme")->fetchAll(PDO::FETCH_COLUMN);
$employee_types = $pdo->query("SELECT DISTINCT employee_type FROM salary_structures WHERE employee_type IS NOT NULL AND employee_type <> '' ORDER BY employee_type")->fetchAll(PDO::FETCH_COLUMN);

// Fetch departments for select
$departmentss = $pdo->query("SELECT name FROM departments ORDER BY name")->fetchAll(PDO::FETCH_COLUMN);

// Handle CSV Bulk Upload
if (isset($_POST['upload_csv']) && isset($_FILES['csv_file'])) {
    $file = $_FILES['csv_file']['tmp_name'];
    $handle = fopen($file, 'r');
    if (!$handle) {
        echo "<script>alert('Cannot open CSV file'); window.location.href='employee_list.php';</script>";
        echo "<noscript><meta http-equiv='refresh' content='0;url=employee_list.php'></noscript>";
        exit;
    }
    $header = fgetcsv($handle);
    if (!$header) {
        fclose($handle);
        echo "<script>alert('CSV header missing'); window.location.href='employee_list.php';</script>";
        echo "<noscript><meta http-equiv='refresh' content='0;url=employee_list.php'></noscript>";
        exit;
    }

    // Header helpers
    $norm = function($s) { return strtolower(trim(preg_replace('/\s+/', ' ', (string)$s))); };
    $idx = [];
    foreach ($header as $i => $h) { $idx[$norm($h)] = $i; }
    $findIdx = function($variants) use ($idx, $norm) {
        foreach ($variants as $v) { $k = $norm($v); if (isset($idx[$k])) return $idx[$k]; }
        return null;
    };
    $parseCompound = function($val) {
        if (!is_string($val)) return [null, null];
        $s = trim($val);
        if ($s === '') return [null, null];
        if (preg_match('/\((\d{1,2})[\.-\/]?(\d{1,2})\)/', $s, $m)) return [$m[1], ltrim($m[2], '0') === '' ? '0' : ltrim($m[2], '0')];
        if (preg_match('/^(\d{1,2})\s*[\.-\/]\s*(\d{1,2})$/', $s, $m)) return [$m[1], ltrim($m[2], '0') === '' ? '0' : ltrim($m[2], '0')];
        return [null, null];
    };

    // Column indices (flexible)
    $iEmpNo   = $findIdx(['Employee No','employee_no','Emp Code','Emp No','Staff No','Staff Number']);
    $iFull    = $findIdx(['Full Name','full_name','Name']);
    $iEmail   = $findIdx(['Email','email']);
    $iPhone   = $findIdx(['Phone','phone']);
    $iGender  = $findIdx(['Gender','gender']);
    $iJob     = $findIdx(['Job Title','job_title']);
    $iEmpType = $findIdx(['Employment Type','employment_type','Employee Type']);
    $iGrade   = $findIdx(['grade_level','Grade Level']);
    $iStep    = $findIdx(['grade_step','Grade Step','step']);
    $iDept    = $findIdx(['DEPARTMENT','Department','department']);
    $iJoined  = $findIdx(['date_joined','Date Joined']);
    $iBank    = $findIdx(['bank_name','Bank Name']);
    $iAcct    = $findIdx(['account_number','Account Number']);
    $iPension = $findIdx(['pension_number','Pension Number']);
    $iNHIS    = $findIdx(['nhis_number','NHIS Number']);
    $iScheme  = $findIdx(['Scheme','scheme']);
    $iStructType = $findIdx(['employee_type','Employee Type']);
    $iChams   = $findIdx(['CHAMS No.','CHAMS No','chams_no','Chams No.','Chams No']);
    $iCLevel  = $findIdx(['c_level','cGrade_level','C Grade Level']);
    $iCStep   = $findIdx(['c_step','cGrade_step','C Grade Step']);
    $iPayGrade = $findIdx(['Pay Grade']);
    $iUTH      = $findIdx(['UTH PAY GRADE']);

    $uniqueNos = [];

    // Detect schema support for c_level/c_step
    $dbName = $pdo->query("SELECT DATABASE()")->fetchColumn();
    $colExists = function($col) use ($pdo, $dbName) {
        $q = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'employees' AND COLUMN_NAME = ?");
        $q->execute([$dbName, $col]);
        return $q->fetchColumn() > 0;
    };
    $hasCLevel = $colExists('c_level');
    $hasCStep  = $colExists('c_step');

    // Prepare INSERT with ON DUPLICATE UPDATE
    $sql = "INSERT INTO employees 
        (employee_no, full_name, email, phone, gender, job_title, employment_type, grade_level, step, department, date_joined, bank_name, account_number, pension_number, nhis_number, chams_no, scheme, employee_type, password";
    if ($hasCLevel) $sql .= ", c_level";
    if ($hasCStep)  $sql .= ", c_step";
    $sql .= ") VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?";
    if ($hasCLevel) $sql .= ",?";
    if ($hasCStep)  $sql .= ",?";
    $sql .= ") ON DUPLICATE KEY UPDATE 
        full_name=VALUES(full_name), email=VALUES(email), phone=VALUES(phone),
        gender=VALUES(gender), job_title=VALUES(job_title), employment_type=VALUES(employment_type),
        grade_level=VALUES(grade_level), step=VALUES(step), department=VALUES(department), date_joined=VALUES(date_joined),
        bank_name=VALUES(bank_name), account_number=VALUES(account_number), pension_number=VALUES(pension_number), nhis_number=VALUES(nhis_number),
        chams_no=VALUES(chams_no), scheme=VALUES(scheme), employee_type=VALUES(employee_type)";
    if ($hasCLevel) $sql .= ", c_level=VALUES(c_level)";
    if ($hasCStep)  $sql .= ", c_step=VALUES(c_step)";

    $stmt = $pdo->prepare($sql);
    $defaultPassword = password_hash('12345678', PASSWORD_DEFAULT);

    while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
        // Skip empty rows
        if (!$iEmpNo && (count($data) < 17 || trim($data[0]) === '')) continue;
        $employee_no = $iEmpNo !== null ? trim($data[$iEmpNo]) : trim($data[0]);
        if (!$employee_no || in_array($employee_no, $uniqueNos)) continue;
        $uniqueNos[] = $employee_no;

        // Resolve values from header mapping, fallback to positional indices if needed
        $full_name = $iFull !== null ? trim($data[$iFull]) : (isset($data[1]) ? trim($data[1]) : '');
        $email     = $iEmail !== null ? trim($data[$iEmail]) : (isset($data[2]) ? trim($data[2]) : '');
        $phone     = $iPhone !== null ? trim($data[$iPhone]) : (isset($data[3]) ? trim($data[3]) : '');
        $gender    = $iGender !== null ? trim($data[$iGender]) : (isset($data[4]) ? trim($data[4]) : '');
        $job_title = $iJob !== null ? trim($data[$iJob]) : (isset($data[5]) ? trim($data[5]) : '');
        $employment_type = $iEmpType !== null ? trim($data[$iEmpType]) : (isset($data[6]) ? trim($data[6]) : '');

        // grade_level and step
        $grade_level = $iGrade !== null ? intval($data[$iGrade]) : (isset($data[7]) ? intval($data[7]) : 0);
        $step        = $iStep !== null ? intval($data[$iStep])  : (isset($data[8]) ? intval($data[8]) : 0);

        // If c_level/c_step provided, prefer them; else try to parse compound grades
        $c_level = null; $c_step = null;
        if ($iCLevel !== null && isset($data[$iCLevel])) $c_level = preg_replace('/\D+/', '', trim($data[$iCLevel]));
        if ($iCStep  !== null && isset($data[$iCStep]))  $c_step  = ltrim(preg_replace('/\D+/', '', trim($data[$iCStep])), '0');
        if (($c_level === null || $c_level === '') || ($c_step === null || $c_step === '')) {
            $compound = null;
            if ($iPayGrade !== null && isset($data[$iPayGrade])) $compound = $data[$iPayGrade];
            if (!$compound && $iUTH !== null && isset($data[$iUTH])) $compound = $data[$iUTH];
            if ($compound) {
                list($lv, $st) = $parseCompound($compound);
                if ($c_level === null || $c_level === '') $c_level = $lv;
                if ($c_step === null  || $c_step === '')  $c_step  = $st;
            }
        }
        // If table lacks c_level/c_step, map to grade_level/step
        if (!$hasCLevel && $c_level) $grade_level = intval($c_level);
        if (!$hasCStep  && $c_step)  $step        = intval($c_step);

        $department  = $iDept !== null ? trim($data[$iDept]) : (isset($data[9]) ? trim($data[9]) : '');
        $date_joined = $iJoined !== null ? trim($data[$iJoined]) : (isset($data[10]) ? trim($data[10]) : '');
        $bank_name   = $iBank !== null ? trim($data[$iBank]) : (isset($data[11]) ? trim($data[11]) : '');
        $account_number = $iAcct !== null ? trim($data[$iAcct]) : (isset($data[12]) ? trim($data[12]) : '');
        $pension_number = $iPension !== null ? trim($data[$iPension]) : (isset($data[13]) ? trim($data[13]) : '');
        $nhis_number    = $iNHIS !== null ? trim($data[$iNHIS]) : (isset($data[14]) ? trim($data[14]) : '');
        $scheme      = $iScheme !== null ? trim($data[$iScheme]) : (isset($data[15]) ? trim($data[15]) : '');
        $employee_type = $iStructType !== null ? trim($data[$iStructType]) : (isset($data[16]) ? trim($data[16]) : '');
        $chams_no    = $iChams !== null ? trim($data[$iChams]) : (isset($data[17]) ? trim($data[17]) : '');

        // Build params in the order of INSERT
        $params = [
            $employee_no, $full_name, $email, $phone, $gender, $job_title, $employment_type,
            $grade_level, $step, $department, $date_joined, $bank_name, $account_number,
            $pension_number, $nhis_number, $chams_no, $scheme, $employee_type, $defaultPassword
        ];
        if ($hasCLevel) $params[] = $c_level ? intval($c_level) : null;
        if ($hasCStep)  $params[] = $c_step  ? intval($c_step)  : null;

        $success = $stmt->execute($params);
        if ($success) {
            log_audit($pdo, $user_id, "Added/Updated employee ID $employee_no - BULK UPLOAD");
        }
    }

    fclose($handle);

    // JS Redirect (safe after output)
    echo "<script>window.location.href='employee_list.php';</script>";
    // HTML fallback
    echo "<noscript><meta http-equiv='refresh' content='0;url=employee_list.php'></noscript>";
    exit;
}

// Handle Add/Edit
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['emp_code'])) {
    $id = $_POST['id'] ?? 0;
    $empcode = $_POST['emp_code'];
    $data = [
        'employee_no' => trim($_POST['emp_code']),
        'full_name' => trim($_POST['full_name']),
        'email' => trim($_POST['email']),
        'phone' => trim($_POST['phone']),
        'gender' => $_POST['gender'],
        'job_title' => trim($_POST['job_title']),
        'employment_type' => $_POST['employment_type'],
        'grade_level' => intval($_POST['grade_level']),
        'step' => intval($_POST['step']),
        'c_level' => isset($_POST['c_level']) && $_POST['c_level'] !== '' ? intval($_POST['c_level']) : null,
        'c_step' => isset($_POST['c_step']) && $_POST['c_step'] !== '' ? intval($_POST['c_step']) : null,
        'chams_no' => trim($_POST['chams_no']),
        'department' => trim($_POST['department']),
        'date_joined' => $_POST['date_joined'],
        'bank_name' => trim($_POST['bank_name']),
        'account_number' => trim($_POST['account_number']),
        'pension_number' => trim($_POST['pension_number']),
        'nhis_number' => trim($_POST['nhis_number']),
        'scheme' => $_POST['scheme'],
        'employee_type' => $_POST['employee_type'],
    ];
    if ($id) {
        $sql = "UPDATE employees SET employee_no=:employee_no, full_name=:full_name, email=:email, phone=:phone, gender=:gender, job_title=:job_title, employment_type=:employment_type, grade_level=:grade_level, step=:step, c_level=:c_level, c_step=:c_step, department=:department, date_joined=:date_joined, bank_name=:bank_name, account_number=:account_number, pension_number=:pension_number, nhis_number=:nhis_number, chams_no=:chams_no, scheme=:scheme, employee_type=:employee_type WHERE id=:id";
        $data['id'] = $id;
    } else {
        $sql = "INSERT INTO employees (employee_no, full_name, email, phone, gender, job_title, employment_type, grade_level, step, c_level, c_step, department, date_joined, bank_name, account_number, pension_number, nhis_number, chams_no, scheme, employee_type, password) VALUES (:employee_no, :full_name, :email, :phone, :gender, :job_title, :employment_type, :grade_level, :step, :c_level, :c_step, :department, :date_joined, :bank_name, :account_number, :pension_number, :nhis_number, :chams_no, :scheme, :employee_type, :password)";
        $data['password'] = password_hash('12345678', PASSWORD_DEFAULT);
        
    }
    $stmt = $pdo->prepare($sql);
    if($stmt->execute($data)){
    log_audit($pdo, $user_id, "Added/Updated employee ID $empcode");
    }
    // JS Redirect (safe after output)
    echo "<script>window.location.href='employee_list.php';</script>";
    echo "<noscript><meta http-equiv='refresh' content='0;url=employee_list.php'></noscript>";
    exit;
}


// Handle Delete
if (isset($_POST['delete'])) {
    $id = intval($_POST['id']);

    // Delete dependent rows first
    $pdo->prepare("DELETE FROM allowances WHERE employee_id = ?")->execute([$id]);
    $pdo->prepare("DELETE FROM deductions WHERE employee_id = ?")->execute([$id]);
    $pdo->prepare("DELETE FROM payroll_details WHERE employee_id = ?")->execute([$id]);
    // Add similar lines for any other table with employee_id as FK

    // Now delete the employee
    $pdo->prepare("DELETE FROM employees WHERE id = ?")->execute([$id]);
    log_audit($pdo, $user_id, "Deleted employee ID $id");
    header("Location: employee_list.php");
    exit;
}


// Filtering
$where = "1=1";
$params = [];
if (!empty($_GET['department'])) {
    $where .= " AND department = ?";
    $params[] = $_GET['department'];
}
if (!empty($_GET['grade_level'])) {
    $where .= " AND grade_level = ?";
    $params[] = $_GET['grade_level'];
}
$employees = $pdo->prepare("SELECT * FROM employees WHERE $where ORDER BY full_name");
$employees->execute($params);
$employees = $employees->fetchAll(PDO::FETCH_ASSOC);

// CSV Export (respects same filters)
if (isset($_GET['export']) && strtolower($_GET['export']) === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="employees_export.csv"');
    $out = fopen('php://output', 'w');
    // Headers including CHAMS and C Grade fields
    fputcsv($out, [
        'Employee No', 'Full Name', 'Email', 'Phone', 'Gender', 'Job Title', 'Employment Type',
        'Grade Level', 'Step', 'C Grade Level', 'C Grade Step', 'Department', 'Date Joined',
        'Bank Name', 'Account Number', 'Pension Number', 'NHIS Number', 'CHAMS No.', 'Scheme', 'Employee Type'
    ]);
    foreach ($employees as $emp) {
        fputcsv($out, [
            $emp['employee_no'] ?? '',
            $emp['full_name'] ?? '',
            $emp['email'] ?? '',
            $emp['phone'] ?? '',
            $emp['gender'] ?? '',
            $emp['job_title'] ?? '',
            $emp['employment_type'] ?? '',
            $emp['grade_level'] ?? '',
            $emp['step'] ?? '',
            $emp['c_level'] ?? '',
            $emp['c_step'] ?? '',
            $emp['department'] ?? '',
            $emp['date_joined'] ?? '',
            $emp['bank_name'] ?? '',
            $emp['account_number'] ?? '',
            $emp['pension_number'] ?? '',
            $emp['nhis_number'] ?? '',
            $emp['chams_no'] ?? '',
            $emp['scheme'] ?? '',
            $emp['employee_type'] ?? ''
        ]);
    }
    fclose($out);
    exit;
}

?>
<div class="page-container">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h4>Employee List</h4>
        <div>
            <!-- Step Increment Button -->
            <form method="POST" style="display: inline-block;" 
                  onsubmit="return confirm('Are you sure you want to increment steps for all eligible employees? This action cannot be undone.');">
                <button type="submit" name="increment_steps" class="btn btn-warning me-2">
                    <i class="fas fa-arrow-up"></i> Increment Employee Steps
                </button>
            </form>
            <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#uploadCsvModal">Bulk Upload (CSV)</button>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#employeeModal" onclick="resetForm()">Add Employee</button>
        </div>
    </div>
    
    <?php
    // Display step increment result message if available
    if (isset($_SESSION['step_increment_result'])) {
        $result = $_SESSION['step_increment_result'];
        $alertClass = $result['success'] ? 'alert-success' : 'alert-danger';
        echo "<div class='alert {$alertClass} alert-dismissible fade show' role='alert'>
                {$result['message']}
                <button type='button' class='btn-close' data-bs-dismiss='alert'></button>
              </div>";
        unset($_SESSION['step_increment_result']);
    }
    ?>

    <!-- Bulk Upload Modal -->
    <div class="modal fade" id="uploadCsvModal" tabindex="-1">
        <div class="modal-dialog">
            <form method="POST" enctype="multipart/form-data" class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Bulk Employee Upload (CSV)</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="file" name="csv_file" class="form-control" accept=".csv" required>
                </div>
                <div class="modal-footer">
                    <button type="submit" name="upload_csv" class="btn btn-success">Upload</button>
                    <a href="sample_employees.csv" class="btn btn-link">Download Sample CSV</a>
                </div>
            </form>
        </div>
    </div>

    <!-- Filter Form -->
    <form method="GET" class="row g-2 mb-3">
        <div class="col-md-3">
            <select name="department" class="form-select" onchange="this.form.submit()">
                <option value="">All Departments</option>
                <?php foreach ($departments as $d): ?>
                <option value="<?= htmlspecialchars($d) ?>" <?= isset($_GET['department']) && $_GET['department'] === $d ? 'selected' : '' ?>><?= htmlspecialchars($d) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-3">
            <select name="grade_level" class="form-select" onchange="this.form.submit()">
                <option value="">All Grades</option>
                <?php foreach ($grades as $g): ?>
                <option value="<?= $g ?>" <?= isset($_GET['grade_level']) && $_GET['grade_level'] == $g ? 'selected' : '' ?>><?= $g ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-auto">
            <button class="btn btn-outline-secondary" type="submit">Filter</button>
        </div>
    </form>
<table id="employeeTable" class="table table-bordered table-striped">
    <thead>
        <tr>
            <th>S/N</th>
            <th>Full Name</th>
            <th>Emp Code</th>
            <th>Job Title</th>
            <th>Dept</th>
            <th>Level</th>
            <th>Step</th>
            <th>Last Step Increment</th>
            <th>Scheme</th>
            <th>Emp Type</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($employees as $i => $emp): ?>
        <tr>
            <td><?= $i + 1 ?></td>
            <td><?= htmlspecialchars($emp['full_name']) ?></td>
            <td><?= htmlspecialchars($emp['employee_no']) ?></td>
            <td><?= htmlspecialchars($emp['job_title']) ?></td>
            <td><?= htmlspecialchars($emp['department']) ?></td>
            <td><?= $emp['grade_level'] ?></td>
            <td><?= $emp['step'] ?></td>
            <td><?= !empty($emp['last_step_increased_at']) ? htmlspecialchars($emp['last_step_increased_at']) : 'Never' ?></td>
            <td><?= htmlspecialchars($emp['scheme']) ?></td>
            <td><?= htmlspecialchars($emp['employee_type']) ?></td>
            <td>
                <button class="btn btn-sm btn-warning" onclick='editEmployee(<?= json_encode($emp) ?>)'>Edit</button>
                <form method="POST" style="display:inline-block">
                    <input type="hidden" name="id" value="<?= $emp['id'] ?>">
                    <button class="btn btn-sm btn-danger" name="delete" onclick="return confirm('Are you sure?')">Delete</button>
                </form>
            </td>
        </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Modal -->
<div class="modal fade" id="employeeModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <form method="POST" class="p-3" id="employeeForm">
        <div class="modal-header">
          <h5 class="modal-title" id="modalTitle">Add New Employee</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body row g-2">
          <input type="hidden" name="id" id="id">
          <div class="col-md-6"><input type="text" name="emp_code" id="emp_code" class="form-control" placeholder="Employee Code" required></div>
          <div class="col-md-6"><input type="text" name="full_name" id="full_name" class="form-control" placeholder="Full Name" required></div>
          <div class="col-md-6"><input type="text" name="chams_no" id="chams_no" class="form-control" placeholder="CHAMS No."></div>
          <div class="col-md-6"><input type="email" name="email" id="email" class="form-control" placeholder="Email"></div>
          <div class="col-md-6"><input type="text" name="phone" id="phone" class="form-control" placeholder="Phone"></div>
          <div class="col-md-6">
            <select name="gender" id="gender" class="form-select">
              <option value="">Gender</option>
              <option value="Male">Male</option>
              <option value="Female">Female</option>
              <option value="Other">Other</option>
            </select>
          </div>
          <div class="col-md-6"><input type="text" name="job_title" id="job_title" class="form-control" placeholder="Job Title"></div>
          <div class="col-md-6">
            <select name="employment_type" id="employment_type" class="form-select">
              <option value="">Employment Type</option>
              <option value="Full-Time">Full-Time</option>
              <option value="Contract">Contract</option>
              <option value="Intern">Intern</option> 
              <option value="Ad-Hoc">Ad-Hoc</option>
            </select>
          </div>
          <div class="col-md-3"><input type="number" name="grade_level" id="grade_level" class="form-control" placeholder="Grade Level"></div>
          <div class="col-md-3"><input type="number" name="step" id="step" class="form-control" placeholder="Step"></div>
          <div class="col-md-3"><input type="number" name="c_level" id="c_level" class="form-control" placeholder="C Grade Level"></div>
          <div class="col-md-3"><input type="number" name="c_step" id="c_step" class="form-control" placeholder="C Grade Step"></div>
          <div class="col-md-6">
            <select name="department" id="department" class="form-select">
              <option value="">Select Department</option>
              <?php foreach ($departments as $depts): ?>
                <option value="<?= htmlspecialchars($depts) ?>"><?= htmlspecialchars($depts) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-6">
            <select name="scheme" id="scheme" class="form-select">
              <option value="">Select Scheme</option>
              <?php foreach ($schemes as $schm): ?>
                <option value="<?= htmlspecialchars($schm) ?>"><?= htmlspecialchars($schm) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-6">
            <select name="employee_type" id="employee_type" class="form-select">
              <option value="">Select Employee Type</option>
              <?php foreach ($employee_types as $emp): ?>
                <option value="<?= htmlspecialchars($emp) ?>"><?= htmlspecialchars($emp) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-6"><input type="date" name="date_joined" id="date_joined" class="form-control"></div>
          <div class="col-md-6"><input type="text" name="bank_name" id="bank_name" class="form-control" placeholder="Bank Name"></div>
          <div class="col-md-6"><input type="text" name="account_number" id="account_number" class="form-control" placeholder="Account Number"></div>
          <div class="col-md-6"><input type="text" name="pension_number" id="pension_number" class="form-control" placeholder="Pension Number"></div>
          <div class="col-md-6"><input type="text" name="nhis_number" id="nhis_number" class="form-control" placeholder="NHIS Number"></div>
        </div>
        <div class="modal-footer">
          <button type="submit" class="btn btn-success">Save</button>
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        </div>
      </form>
    </div>
  </div>
</div>


		    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
		    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
<script>
$(document).ready(function() {
  $('#employeeTable').DataTable();
});

// Open modal for adding
function resetForm() {
  $('#modalTitle').text('Add New Employee');
  $('#employeeForm')[0].reset();
  $('#id').val('');
  $('#employeeModal').modal('show');
}

// Open modal for editing
function editEmployee(emp) {
  $('#modalTitle').text('Edit Employee');
  $('#id').val(emp.id);
  $('#emp_code').val(emp.employee_no);
  $('#full_name').val(emp.full_name);
  $('#email').val(emp.email);
  $('#phone').val(emp.phone);
  $('#gender').val(emp.gender);
  $('#job_title').val(emp.job_title); 
  $('#employment_type').val(emp.employment_type);
  $('#grade_level').val(emp.grade_level);
  $('#step').val(emp.step);
  $('#chams_no').val(emp.chams_no);
  $('#c_level').val(emp.c_level);
  $('#c_step').val(emp.c_step);
  $('#department').val(emp.department);
  $('#date_joined').val(emp.date_joined);
  $('#bank_name').val(emp.bank_name);
  $('#account_number').val(emp.account_number);
  $('#pension_number').val(emp.pension_number);
  $('#nhis_number').val(emp.nhis_number);
  $('#employeeModal').modal('show');
}
</script>


</div>
