<?php
require_once '../database/config.php';
session_start();

$user_id = $_SESSION['user_id'] ?? null;
// Handle Add/Edit POST
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['employee_id'])) {
    $employee_id = intval($_POST['employee_id']);
    $type = $_POST['type'];
    $amount = floatval($_POST['amount']);
    $deduction_date = $_POST['deduction_date'];
    $recurring = isset($_POST['recurring']) ? 1 : 0;
    $description = $_POST['description'] ?? null;
    $edit_id = isset($_POST['edit_id']) ? intval($_POST['edit_id']) : 0;

    try { 
        if ($edit_id > 0) {
            $stmt = $pdo->prepare("UPDATE deductions SET employee_id=?, type=?, amount=?, deduction_date=?, recurring=?, description=? WHERE id=?");
            $stmt->execute([$employee_id, $type, $amount, $deduction_date, $recurring, $description, $edit_id]);
        } else {
            $stmt = $pdo->prepare("INSERT INTO deductions (employee_id, type, amount, deduction_date, recurring, description) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$employee_id, $type, $amount, $deduction_date, $recurring, $description]);
        }
        $_SESSION['flash'] = "$type Deduction saved successfully.";
    } catch (PDOException $e) {
        $_SESSION['flash'] = "Error: Possible duplicate entry or invalid data.";
    }

    log_audit($pdo, $user_id, $_SESSION['flash']);
    header("Location: deductions.php");
    exit;
}

 
// Handle Bulk Deduction POST
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['department_deduction'])) {
    $department = $_POST['department'];
    $type = $_POST['type'];
    $amount = floatval($_POST['amount']);
    $deduction_date = $_POST['deduction_date'];
    $recurring = isset($_POST['recurring']) ? 1 : 0;
    $description = $_POST['description'] ?? null;

    $stmt = $pdo->prepare("SELECT id FROM employees WHERE department = ?");
    $stmt->execute([$department]);
    $employees = $stmt->fetchAll(PDO::FETCH_COLUMN);

    foreach ($employees as $emp_id) {
        try {
            $stmt = $pdo->prepare("INSERT INTO deductions (employee_id, type, amount, deduction_date, recurring, description) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$emp_id, $type, $amount, $deduction_date, $recurring, $description]);
        } catch (PDOException $e) {
            // Skip duplicates silently
        }
    }
    $_SESSION['flash'] = "Bulk deductions applied to $department department.";
    
    log_audit($pdo, $user_id, $_SESSION['flash']);
    header("Location: deductions.php");
    exit;
}

// Handle CSV Preview
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csv_preview'])) {
    header('Content-Type: application/json');

    $type = trim($_POST['type'] ?? '');
    $deduction_date = $_POST['deduction_date'] ?? '';
    $recurring = isset($_POST['recurring']) ? 1 : 0;

    if (empty($type) || empty($deduction_date) || !isset($_FILES['csv_file'])) {
        echo json_encode(['status' => 'error', 'message' => 'Missing required fields or CSV file']);
        exit;
    }

    if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to upload CSV file']);
        exit;
    }

    $tmp = $_FILES['csv_file']['tmp_name'];

    $rows = [];
    $summary = [
        'total' => 0,
        'valid' => 0,
        'duplicates' => 0,
        'not_found' => 0,
        'invalid' => 0
    ];

    $stmtFindEmp = $pdo->prepare("SELECT id, full_name, department FROM employees WHERE employee_no = ?");
    $stmtCheckDup = $pdo->prepare("SELECT id FROM deductions WHERE employee_id=? AND type=? AND deduction_date=? AND recurring=? AND amount=?");

    if (($handle = fopen($tmp, 'r')) !== false) {
        $rowNum = 0;
        while (($row = fgetcsv($handle)) !== false) {
            $rowNum++;
            $summary['total']++;
            if ($rowNum === 1) {
                $c0 = strtolower(trim($row[0] ?? ''));
                $c1 = strtolower(trim($row[1] ?? ''));
                if ($c0 === 'employee_code' || $c0 === 'employee_no' || $c1 === 'amount') {
                    $summary['total']--; // header row
                    continue;
                }
            }

            if (!isset($row[0]) || !isset($row[1])) {
                $summary['invalid']++;
                $rows[] = [
                    'code' => $row[0] ?? '',
                    'amount' => $row[1] ?? '',
                    'status' => 'invalid',
                    'reason' => 'Missing code or amount'
                ];
                continue;
            }

            $code = trim($row[0]);
            $amount = floatval($row[1]);
            if ($code === '' || $amount <= 0) {
                $summary['invalid']++;
                $rows[] = [
                    'code' => $code,
                    'amount' => $amount,
                    'status' => 'invalid',
                    'reason' => 'Empty code or non-positive amount'
                ];
                continue;
            }

            $stmtFindEmp->execute([$code]);
            $emp = $stmtFindEmp->fetch(PDO::FETCH_ASSOC);
            if (!$emp) {
                $summary['not_found']++;
                $rows[] = [
                    'code' => $code,
                    'amount' => $amount,
                    'status' => 'not_found',
                    'reason' => 'Employee not found'
                ];
                continue;
            }

            $stmtCheckDup->execute([$emp['id'], $type, $deduction_date, $recurring, $amount]);
            $dup = $stmtCheckDup->fetch() ? true : false;
            if ($dup) {
                $summary['duplicates']++;
            } else {
                $summary['valid']++;
            }
            $rows[] = [
                'code' => $code,
                'amount' => $amount,
                'name' => $emp['full_name'],
                'department' => $emp['department'],
                'employee_id' => $emp['id'],
                'status' => $dup ? 'duplicate' : 'valid',
                'reason' => $dup ? 'Duplicate exists' : null
            ];
        }
        fclose($handle);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Could not read uploaded CSV file']);
        exit;
    }

    echo json_encode([
        'status' => 'success',
        'summary' => $summary,
        'rows' => $rows
    ]);
    exit;
}

// Handle CSV Deduction Upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csv_deduction'])) {
    $type = trim($_POST['type'] ?? '');
    $deduction_date = $_POST['deduction_date'] ?? '';
    $recurring = isset($_POST['recurring']) ? 1 : 0;
    $description = $_POST['description'] ?? null;

    if (empty($type) || empty($deduction_date) || !isset($_FILES['csv_file'])) {
        $_SESSION['flash'] = "Error: Missing required fields or CSV file.";
        log_audit($pdo, $user_id, $_SESSION['flash']);
        header("Location: deductions.php");
        exit;
    }

    if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        $_SESSION['flash'] = "Error: Failed to upload CSV file.";
        log_audit($pdo, $user_id, $_SESSION['flash']);
        header("Location: deductions.php");
        exit;
    }

    $inserted = 0; $duplicates = 0; $not_found = 0; $invalid_rows = 0; $not_found_codes = []; $dept_inserts = [];
    $tmp = $_FILES['csv_file']['tmp_name'];

    if (($handle = fopen($tmp, 'r')) !== false) {
        $rowNum = 0;
        $stmtFindEmp = $pdo->prepare("SELECT id, full_name, department FROM employees WHERE employee_no = ?");
        $stmtCheckDup = $pdo->prepare("SELECT id FROM deductions WHERE employee_id=? AND type=? AND deduction_date=? AND recurring=? AND amount=?");
        $stmtInsert = $pdo->prepare("INSERT INTO deductions (employee_id, type, amount, deduction_date, recurring, description) VALUES (?, ?, ?, ?, ?, ?)");

        while (($row = fgetcsv($handle)) !== false) {
            $rowNum++;
            if ($rowNum === 1) {
                $c0 = strtolower(trim($row[0] ?? ''));
                $c1 = strtolower(trim($row[1] ?? ''));
                if ($c0 === 'employee_code' || $c0 === 'employee_no' || $c1 === 'amount') {
                    // Skip header row
                    continue;
                }
            }

            if (!isset($row[0]) || !isset($row[1])) { $invalid_rows++; continue; }

            $code = trim($row[0]);
            $amount = floatval($row[1]);
            if ($code === '' || $amount <= 0) { $invalid_rows++; continue; }

            $stmtFindEmp->execute([$code]);
            $emp = $stmtFindEmp->fetch(PDO::FETCH_ASSOC);
            if (!$emp) { $not_found++; if (count($not_found_codes) < 10) { $not_found_codes[] = $code; } continue; }

            $stmtCheckDup->execute([$emp['id'], $type, $deduction_date, $recurring, $amount]);
            if ($stmtCheckDup->fetch()) { $duplicates++; continue; }

            try {
                $stmtInsert->execute([$emp['id'], $type, $amount, $deduction_date, $recurring, $description]);
                $inserted++;
                $dept = $emp['department'] ?: 'Unknown';
                $dept_inserts[$dept] = ($dept_inserts[$dept] ?? 0) + 1;
            } catch (PDOException $e) {
                $duplicates++; // treat as duplicate or invalid data
            }
        }
        fclose($handle);

        $deptSummary = '';
        if (!empty($dept_inserts)) {
            $parts = [];
            foreach ($dept_inserts as $dept => $cnt) { $parts[] = $dept . ':' . $cnt; }
            $deptSummary = ' | Departments inserted: ' . implode(', ', $parts);
        }

        $_SESSION['flash'] = "CSV upload complete. Inserted: $inserted, Duplicates: $duplicates, Not found: $not_found, Invalid rows: $invalid_rows" . $deptSummary . (count($not_found_codes) ? ". Missing codes: " . implode(', ', $not_found_codes) : "");
    } else {
        $_SESSION['flash'] = "Error: Could not read uploaded CSV file.";
    }

    log_audit($pdo, $user_id, $_SESSION['flash']);
    header("Location: deductions.php");
    exit;
}

// Handle Type Add/Delete
if (isset($_POST['add_type'])) {
    $new_type = trim($_POST['new_type']);
    if (!empty($new_type)) {
        try {
            $stmt = $pdo->prepare("INSERT INTO deduction_types (name) VALUES (?)");
            $stmt->execute([$new_type]);
            $_SESSION['flash'] = "New deduction type added.";
        } catch (PDOException $e) {
            $_SESSION['flash'] = "Error: Could not add type (may already exist).";
        }
    }
    log_audit($pdo, $user_id, "Added Deduction Type: $new_type");
    header("Location: deductions.php");
    exit;
}

if (isset($_GET['delete_type'])) {
    $id = intval($_GET['delete_type']);
    $pdo->prepare("DELETE FROM deduction_types WHERE id = ?")->execute([$id]);
    $_SESSION['flash'] = "Deduction type deleted.";
    log_audit($pdo, $user_id, "Deleted Deduction Type: $id");
    header("Location: deductions.php");
    exit;
}

// Handle Deduction Delete
if (isset($_GET['delete_id'])) {
    $id = intval($_GET['delete_id']);
    $pdo->prepare("DELETE FROM deductions WHERE id = ?")->execute([$id]);
    $_SESSION['flash'] = "Deduction deleted.";
    log_audit($pdo, $user_id, "Deleted Deduction: $id");
    header("Location: deductions.php");
    exit;
}

$employees = $pdo->query("SELECT id, full_name FROM employees ORDER BY full_name")->fetchAll(PDO::FETCH_ASSOC);
$types = $pdo->query("SELECT * FROM deduction_types ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

$filter_emp_code = $_GET['employee_code'] ?? '';
if ($filter_emp_code) {
    $stmt = $pdo->prepare("SELECT d.*, e.full_name, e.employee_no AS employee_code FROM deductions d JOIN employees e ON d.employee_id = e.id WHERE e.employee_no = ?");
    $stmt->execute([$filter_emp_code]);
} else {
    $stmt = $pdo->query("SELECT d.*, e.full_name, e.employee_no AS employee_code FROM deductions d JOIN employees e ON d.employee_id = e.id");
}
$deductions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$departments = $pdo->query("SELECT DISTINCT department FROM employees WHERE department IS NOT NULL ORDER BY department")->fetchAll(PDO::FETCH_COLUMN);


?> 
<?php include 'header.php'?>
<div class="container py-4">
    <?php if (!empty($_SESSION['flash'])): ?>
        <div class="alert alert-info alert-dismissible fade show" role="alert">
            <?= $_SESSION['flash']; unset($_SESSION['flash']); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>
<!-- Bulk Deduction Modal -->
<div class="modal fade" id="bulkDeductionModal" tabindex="-1" aria-labelledby="bulkDeductionModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="bulkDeductionModalLabel">Apply Bulk Deduction</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="department_deduction" value="1">
        <div class="mb-3">
          <label class="form-label">Department</label>
          <select name="department" class="form-select" required>
            <option value="">-- Select Department --</option>
            <?php foreach ($departments as $dept): ?>
              <option value="<?= htmlspecialchars($dept) ?>"><?= htmlspecialchars($dept) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-3">
          <label class="form-label">Deduction Type</label>
          <select name="type" class="form-select" required>
            <option value="">-- Select Type --</option>
            <?php foreach ($types as $t): ?>
              <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-3">
          <label class="form-label">Amount (₦)</label>
          <input type="number" name="amount" step="0.01" class="form-control" required>
        </div>
        <div class="mb-3">
          <label class="form-label">Deduction Date</label>
          <input type="date" name="deduction_date" class="form-control" required>
        </div>
        <div class="form-check mb-3">
          <input type="checkbox" class="form-check-input" name="recurring" id="bulk_recurring">
          <label class="form-check-label" for="bulk_recurring">Recurring</label>
        </div>
        <div class="mb-3">
          <label class="form-label">Description (Optional)</label>
          <textarea class="form-control" name="description" rows="2"></textarea>
        </div>
      </div>
      <div class="modal-footer">
        <button type="submit" class="btn btn-success">Apply Deduction</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<!-- CSV Deduction Upload Modal -->
<div class="modal fade" id="csvDeductionModal" tabindex="-1" aria-labelledby="csvDeductionModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" enctype="multipart/form-data" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="csvDeductionModalLabel">Upload CSV Deductions</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="csv_deduction" value="1">
        <div class="mb-3">
          <label class="form-label">Deduction Type</label>
          <select name="type" class="form-select" required>
            <option value="">-- Select Type --</option>
            <?php foreach ($types as $t): ?>
              <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-3">
          <label class="form-label">Deduction Date</label>
          <input type="date" name="deduction_date" class="form-control" required>
        </div>
        <div class="form-check mb-3">
          <input type="checkbox" class="form-check-input" name="recurring" id="csv_recurring">
          <label class="form-check-label" for="csv_recurring">Recurring</label>
        </div>
        <div class="mb-3">
          <label class="form-label">CSV File</label>
          <input type="file" name="csv_file" accept=".csv" class="form-control" required>
          <small class="text-muted">CSV columns: employee_code, amount. Department is auto-detected. <br><b>Make sure file is saved as .csv</b></small>
        </div>
        <div class="mb-3">
          <label class="form-label">Description (Optional)</label>
          <textarea class="form-control" name="description" rows="2"></textarea>
        </div>
        <div id="csvPreviewContainer" class="mt-3"></div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-info" id="previewCsvBtn" onclick="previewCsv()">Preview</button>
        <button type="submit" class="btn btn-success">Upload</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<!-- Button to trigger modal -->
<div class="d-flex justify-content-end mb-3">
  <button class="btn btn-outline-primary me-2" data-bs-toggle="modal" data-bs-target="#bulkDeductionModal">Bulk Deduction</button>
  <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#csvDeductionModal">CSV Upload</button>
</div>

    <div class="d-flex justify-content-between align-items-center mb-3">
        <h4>Deductions</h4>
        <div>
            <button class="btn btn-secondary" data-bs-toggle="modal" data-bs-target="#typeModal">Manage Types</button>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#deductionModal" onclick="openAddModal()">Add Deduction</button>
        </div>
    </div>

<form class="row g-3 mb-3" method="get">
    <div class="col-md-4">
        <label for="employee_code" class="form-label">Filter by Employee Code</label>
        <input type="text" name="employee_code" id="employee_code_filter" class="form-control"
               value="<?= isset($_GET['employee_code']) ? htmlspecialchars($_GET['employee_code']) : '' ?>"
               placeholder="Enter Employee Code (e.g., EMP001)">
    </div>
    <div class="col-md-2 align-self-end">
        <button class="btn btn-primary btn-sm" type="submit">Filter</button>
    </div>
</form>


    <table id="deductionTable" class="table table-bordered table-striped">
        <thead>
            <tr>
                <th>ID</th>
                <th>Employee</th>
                <th>Type</th>
                <th>Amount (₦)</th>
                <th>Date</th>
                <th>Recurring</th>
                <th>Description</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($deductions as $row): ?>
                <tr>
                    <td><?= $row['id'] ?></td>
                    <td><?= htmlspecialchars($row['full_name']) ?></td>
                    <td><?= $row['type'] ?></td>
                    <td><?= number_format($row['amount'], 2) ?></td>
                    <td><?= $row['deduction_date'] ?></td>
                    <td><?= $row['recurring'] ? 'Yes' : 'No' ?></td>
                    <td><?= htmlspecialchars($row['description']) ?></td>
                    <td>
                        <button class="btn btn-sm btn-warning" onclick='openEditModal(<?= json_encode($row) ?>)'>Edit</button>
                        <a href="deductions.php?delete_id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this deduction?')">Delete</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- Type Management Modal -->
<div class="modal fade" id="typeModal" tabindex="-1" aria-labelledby="typeModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="typeModalLabel">Manage Deduction Types</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <label for="new_type" class="form-label">New Type</label>
          <input type="text" name="new_type" id="new_type" class="form-control" required>
          <input type="hidden" name="add_type" value="1">
        </div>
        <hr>
        <ul class="list-group">
          <?php foreach ($types as $t): ?>
            <li class="list-group-item d-flex justify-content-between align-items-center">
              <?= htmlspecialchars($t['name']) ?>
              <a href="?delete_type=<?= $t['id'] ?>" onclick="return confirm('Delete this type?')" class="btn btn-sm btn-danger">Delete</a>
            </li>
          <?php endforeach; ?>
        </ul>
      </div>
      <div class="modal-footer">
        <button type="submit" class="btn btn-success">Add</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </form>
  </div>
</div>
<!-- Deduction Modal -->
<div class="modal fade" id="deductionModal" tabindex="-1" aria-labelledby="deductionModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="deductionModalLabel">Add/Edit Deduction</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="edit_id" id="edit_id">
        <div class="mb-3">
            <label>Employee Code</label>
            <input type="text" name="employee_code" id="employee_code" class="form-control" required onblur="fetchEmployeeName(this.value)">
            <input type="hidden" name="employee_id" id="employee_id_input">
        </div>
        <div class="mb-3">
            <label>Employee Name</label>
            <input type="text" id="employee_name_display" class="form-control" readonly>
        </div>

        <div class="mb-3">
            <label>Type</label>
            <select name="type" id="type" class="form-select" required>
                <option value="">-- Select Type --</option>
                <?php foreach ($types as $t): ?>
                    <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-3">
            <label>Amount (₦)</label>
            <input type="number" step="0.01" name="amount" id="amount" class="form-control" required>
        </div>
        <div class="mb-3">
            <label>Date</label>
            <input type="date" name="deduction_date" id="deduction_date" class="form-control" required>
        </div>
        <div class="form-check mb-3">
            <input class="form-check-input" type="checkbox" name="recurring" id="recurring">
            <label class="form-check-label" for="recurring">Recurring</label>
        </div>
        <div class="mb-3">
            <label>Description (Optional)</label>
            <textarea class="form-control" name="description" id="description" rows="2"></textarea>
        </div>
      </div>
      <div class="modal-footer">
        <button type="submit" class="btn btn-success">Save</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<!-- Scripts -->
<script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>

<script>
$(document).ready(function () {
    $('#deductionTable').DataTable();
});

function fetchEmployeeName(code) {
    if (code.trim() === "") {
        $('#employee_name_display').val('');
        $('#employee_id_input').val('');
        return;
    }

    $.post('fetch_employee_name.php', { code: code }, function(res) {
        if (res.status === 'success') {
            $('#employee_name_display').val(res.name);
            $('#employee_id_input').val(res.id);
        } else {
            $('#employee_name_display').val('Not found');
            $('#employee_id_input').val('');
        }
    }, 'json');
}


function openAddModal() {
    $('#deductionModalLabel').text("Add Deduction");
    $('#edit_id').val('');
    $('#employee_id_input').val('');
    $('#type').val('');
    $('#amount').val('');
    $('#deduction_date').val('');
    $('#recurring').prop('checked', false);
    $('#description').val('');
}

function openEditModal(data) {
    $('#deductionModalLabel').text("Edit Deduction");
    $('#edit_id').val(data.id);
    $('#employee_code').val(data.employee_code || ''); // set code if available
    $('#type').val(data.type);
    $('#amount').val(data.amount);
    $('#deduction_date').val(data.deduction_date);
    $('#recurring').prop('checked', data.recurring == 1);
    $('#description').val(data.description);

    if (data.employee_code) {
        fetchEmployeeName(data.employee_code);
    } else {
        $('#employee_name_display').val('Not found');
        $('#employee_id_input').val('');
    }

    new bootstrap.Modal(document.getElementById('deductionModal')).show();
}

function escapeHtml(s){ return $('<div/>').text(s || '').html(); }

function previewCsv() {
    var $form = $('#csvDeductionModal form');
    var fd = new FormData($form[0]);
    fd.append('csv_preview', '1');

    $('#previewCsvBtn').prop('disabled', true).text('Previewing...');
    $('#csvPreviewContainer').html('');

    $.ajax({
        url: 'deductions.php',
        method: 'POST',
        data: fd,
        processData: false,
        contentType: false,
        dataType: 'json'
    }).done(function(res){
        $('#previewCsvBtn').prop('disabled', false).text('Preview');
        if (!res || res.status !== 'success') {
            var msg = (res && res.message) ? res.message : 'Preview failed';
            $('#csvPreviewContainer').html('<div class="alert alert-danger">'+escapeHtml(msg)+'</div>');
            return;
        }
        var sum = res.summary || { total:0, valid:0, duplicates:0, not_found:0, invalid:0 };
        var html = '';
        html += '<div class="alert alert-info"><strong>Summary</strong>: Total '+sum.total+', Valid '+sum.valid+', Duplicates '+sum.duplicates+', Not found '+sum.not_found+', Invalid '+sum.invalid+'</div>';
        html += '<div class="table-responsive"><table class="table table-sm table-bordered"><thead><tr><th>Employee Code</th><th>Name</th><th>Department</th><th>Amount</th><th>Status</th><th>Reason</th></tr></thead><tbody>';
        (res.rows || []).forEach(function(r){
            html += '<tr>'+
                '<td>'+escapeHtml(r.code)+'</td>'+
                '<td>'+escapeHtml(r.name || '')+'</td>'+
                '<td>'+escapeHtml(r.department || '')+'</td>'+
                '<td>'+((r.amount!=null)? Number(r.amount).toFixed(2) : '')+'</td>'+
                '<td>'+escapeHtml(r.status || '')+'</td>'+
                '<td>'+escapeHtml(r.reason || '')+'</td>'+
            '</tr>';
        });
        html += '</tbody></table></div>';
        $('#csvPreviewContainer').html(html);
    }).fail(function(){
        $('#previewCsvBtn').prop('disabled', false).text('Preview');
        $('#csvPreviewContainer').html('<div class="alert alert-danger">Preview request failed.</div>');
    });
}
</script>

<?php include 'footer.php'?>