<?php
require_once '../database/config.php';
include 'header.php';

// Preview handler: returns JSON analysis of CSV without persisting
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csv_grade_step_preview'])) {
    header('Content-Type: application/json');

    if (!isset($_FILES['csv_file'])) {
        echo json_encode(['status' => 'error', 'message' => 'CSV file is required.']);
        exit;
    }
    if (!is_uploaded_file($_FILES['csv_file']['tmp_name'])) {
        echo json_encode(['status' => 'error', 'message' => 'File upload failed.']);
        exit;
    }

    $fh = fopen($_FILES['csv_file']['tmp_name'], 'r');
    if (!$fh) {
        echo json_encode(['status' => 'error', 'message' => 'Unable to read CSV file.']);
        exit;
    }
    $rows_raw = [];
    while (($row = fgetcsv($fh)) !== false) {
        if ($row === [null] || (count($row) === 1 && trim((string)$row[0]) === '')) { continue; }
        $rows_raw[] = $row;
    }
    fclose($fh);

    if (empty($rows_raw)) {
        echo json_encode(['status' => 'error', 'message' => 'CSV file appears to be empty.']);
        exit;
    }

    // Detect header via known column names in first row
    $first_lower = array_map(function($v){ return strtolower(trim((string)$v)); }, $rows_raw[0]);
    $has_header = in_array('employee no', $first_lower, true) || in_array('employee_no', $first_lower, true) || in_array('grade_level', $first_lower, true) || in_array('step', $first_lower, true);
    if ($has_header) { array_shift($rows_raw); }

    $total = 0; $valid = 0; $invalid = 0; $missing = 0; $no_change = 0; $dept_counts = [];
    $detail_rows = [];

    foreach ($rows_raw as $r) {
        $total++;
        $employee_no = trim((string)($r[0] ?? ''));
        $grade_val = trim((string)($r[1] ?? ''));
        $step_val = trim((string)($r[2] ?? ''));

        if ($employee_no === '' || !is_numeric($grade_val) || !is_numeric($step_val)) {
            $invalid++;
            $detail_rows[] = [
                'employee_no' => $employee_no,
                'grade_level' => $grade_val,
                'step' => $step_val,
                'status' => 'invalid',
                'message' => $employee_no === '' ? 'Missing employee_no' : 'grade_level and step must be numeric'
            ];
            continue;
        }
        $grade_level = (int)$grade_val;
        $step = (int)$step_val;

        // Employee lookup
        $stmt = $pdo->prepare("SELECT id, full_name, department, grade_level AS current_grade, step AS current_step FROM employees WHERE employee_no = ?");
        $stmt->execute([$employee_no]);
        $emp = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$emp) {
            $missing++;
            $detail_rows[] = [
                'employee_no' => $employee_no,
                'grade_level' => $grade_level,
                'step' => $step,
                'status' => 'missing_employee',
                'message' => 'Employee not found'
            ];
            continue;
        }

        $dept_name = $emp['department'] ?? 'Unknown';
        $dept_counts[$dept_name] = ($dept_counts[$dept_name] ?? 0) + 1;

        if ((int)$emp['current_grade'] === $grade_level && (int)$emp['current_step'] === $step) {
            $no_change++;
            $detail_rows[] = [
                'employee_no' => $employee_no,
                'employee_name' => $emp['full_name'],
                'department' => $emp['department'],
                'grade_level' => $grade_level,
                'step' => $step,
                'status' => 'no_change',
                'message' => 'Already at target grade/step'
            ];
            continue;
        }

        $valid++;
        $detail_rows[] = [
            'employee_no' => $employee_no,
            'employee_name' => $emp['full_name'],
            'department' => $emp['department'],
            'grade_level' => $grade_level,
            'step' => $step,
            'status' => 'ok',
            'message' => 'Ready to update'
        ];
    }

    echo json_encode([
        'status' => 'ok',
        'summary' => [
            'total_rows' => $total,
            'valid_rows' => $valid,
            'invalid_rows' => $invalid,
            'missing_employee_rows' => $missing,
            'no_change_rows' => $no_change,
            'departments' => $dept_counts,
        ],
        'rows' => $detail_rows,
    ]);
    exit;
}

// Upload handler: applies updates to employees table
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csv_grade_step_upload'])) {
    if (!isset($_FILES['csv_file'])) {
        $_SESSION['flash'] = 'Upload failed: CSV file is required.';
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='bulk_grade_step_upload.php';</script>";
        exit;
    }
    if (!is_uploaded_file($_FILES['csv_file']['tmp_name'])) {
        $_SESSION['flash'] = 'Upload failed: file not received.';
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='bulk_grade_step_upload.php';</script>";
        exit;
    }

    $updated = 0; $invalid = 0; $missing = 0; $no_change = 0; $dept_counts = [];

    $fh = fopen($_FILES['csv_file']['tmp_name'], 'r');
    $rows_raw = [];
    while (($row = fgetcsv($fh)) !== false) {
        if ($row === [null] || (count($row) === 1 && trim((string)$row[0]) === '')) { continue; }
        $rows_raw[] = $row;
    }
    fclose($fh);

    if (!empty($rows_raw)) {
        $first_lower = array_map(function($v){ return strtolower(trim((string)$v)); }, $rows_raw[0]);
        $has_header = in_array('employee no', $first_lower, true) || in_array('employee_no', $first_lower, true) || in_array('grade_level', $first_lower, true) || in_array('step', $first_lower, true);
        if ($has_header) { array_shift($rows_raw); }
    }

    foreach ($rows_raw as $r) {
        $employee_no = trim((string)($r[0] ?? ''));
        $grade_val = trim((string)($r[1] ?? ''));
        $step_val = trim((string)($r[2] ?? ''));

        if ($employee_no === '' || !is_numeric($grade_val) || !is_numeric($step_val)) { $invalid++; continue; }
        $grade_level = (int)$grade_val; $step = (int)$step_val;

        $stmt = $pdo->prepare('SELECT id, department, grade_level AS current_grade, step AS current_step FROM employees WHERE employee_no = ?');
        $stmt->execute([$employee_no]);
        $emp = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$emp) { $missing++; continue; }

        if ((int)$emp['current_grade'] === $grade_level && (int)$emp['current_step'] === $step) { $no_change++; continue; }

        try {
            $upd = $pdo->prepare('UPDATE employees SET grade_level = ?, step = ? WHERE id = ?');
            $upd->execute([$grade_level, $step, $emp['id']]);
            $updated++;
            $dept_name = $emp['department'] ?? 'Unknown';
            $dept_counts[$dept_name] = ($dept_counts[$dept_name] ?? 0) + 1;
        } catch (PDOException $e) {
            $invalid++;
        }
    }

    $dept_summary = [];
    foreach ($dept_counts as $d => $c) { $dept_summary[] = "$d: $c"; }
    $dept_summary_str = empty($dept_summary) ? 'None' : implode(', ', $dept_summary);

    $_SESSION['flash'] = "Grade/Step updates applied. Updated: $updated, No Change: $no_change, Missing Employees: $missing, Invalid Rows: $invalid. Departments: $dept_summary_str.";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='bulk_grade_step_upload.php';</script>";
    exit;
}
?>

<div class="container py-4">
  <?php if (!empty($_SESSION['flash'])): ?>
    <div class="alert alert-info alert-dismissible fade show" role="alert">
      <?= $_SESSION['flash']; unset($_SESSION['flash']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <div class="d-flex justify-content-between align-items-center mb-3">
    <h4>Bulk Update Grade Level & Step (CSV)</h4>
    <div>
      <a href="allowances.php" class="btn btn-secondary">Back</a>
    </div>
  </div>

  <div class="card">
    <div class="card-body">
      <form method="post" enctype="multipart/form-data" id="gradeStepCsvForm">
        <input type="hidden" name="csv_grade_step_upload" value="1">
        <div class="mb-3">
          <label class="form-label">CSV File</label>
          <input type="file" name="csv_file" class="form-control" accept=".csv,text/csv" required>
          <small class="form-text text-muted">CSV columns: Employee No, grade_level, step</small>
        </div>
        <div id="gradeStepPreviewContainer" class="mt-3"></div>
        <div class="d-flex gap-2">
          <button type="button" class="btn btn-info" onclick="previewGradeStepCsv()">Preview</button>
          <button type="submit" class="btn btn-success">Apply Update</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
function previewGradeStepCsv() {
  var form = document.getElementById('gradeStepCsvForm');
  if (!form) return;
  var fd = new FormData(form);
  fd.delete('csv_grade_step_upload');
  fd.append('csv_grade_step_preview', '1');

  var container = document.getElementById('gradeStepPreviewContainer');
  if (container) {
    container.innerHTML = '<div class="alert alert-info">Generating preview...</div>';
  }

  $.ajax({
    url: 'bulk_grade_step_upload.php',
    method: 'POST',
    data: fd,
    processData: false,
    contentType: false,
    dataType: 'json',
    success: function(res) {
      if (!container) return;
      if (res.status !== 'ok') {
        container.innerHTML = '<div class="alert alert-danger">'+(res.message || 'Preview failed')+'</div>';
        return;
      }
      var s = res.summary || {};
      var deptSummary = s.departments || {};
      var deptList = Object.keys(deptSummary).map(function(d){ return d+': '+deptSummary[d]; }).join(', ');
      var html = '';
      html += '<div class="alert alert-success">';
      html += 'Total: '+(s.total_rows||0)+', Valid: '+(s.valid_rows||0)+', No Change: '+(s.no_change_rows||0)+', Missing: '+(s.missing_employee_rows||0)+', Invalid: '+(s.invalid_rows||0);
      html += '</div>';
      html += '<div><strong>Departments:</strong> '+(deptList || 'None')+'</div>';

      html += '<div class="table-responsive mt-2"><table class="table table-sm table-bordered"><thead><tr><th>Employee No</th><th>Name</th><th>Department</th><th>Current Grade</th><th>Current Step</th><th>New Grade</th><th>New Step</th><th>Status</th><th>Message</th></tr></thead><tbody>';
      (res.rows || []).forEach(function(r){
        html += '<tr>';
        html += '<td>'+(r.employee_no||'')+'</td>';
        html += '<td>'+(r.employee_name||'')+'</td>';
        html += '<td>'+(r.department||'')+'</td>';
        html += '<td>'+(r.current_grade||'')+'</td>';
        html += '<td>'+(r.current_step||'')+'</td>';
        html += '<td>'+(r.grade_level||'')+'</td>';
        html += '<td>'+(r.step||'')+'</td>';
        var badgeClass = (r.status==='ok') ? 'bg-success' : (r.status==='no_change' ? 'bg-warning' : (r.status==='missing_employee' ? 'bg-danger' : 'bg-secondary'));
        html += '<td><span class="badge '+badgeClass+'">'+(r.status||'')+'</span></td>';
        html += '<td>'+(r.message||'')+'</td>';
        html += '</tr>';
      });
      html += '</tbody></table></div>';
      container.innerHTML = html;
    },
    error: function(xhr) {
      if (container) {
        container.innerHTML = '<div class="alert alert-danger">Preview request failed.</div>';
      }
    }
  });
}
</script>

<?php include 'footer.php'; ?>