<?php
require_once '../database/config.php';
if (session_status() === PHP_SESSION_NONE) { session_start(); }

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['preview_chams']) && $_POST['preview_chams'] === '1') {
    header('Content-Type: application/json');

    $use_uploaded = isset($_FILES['csv_file']) && is_uploaded_file($_FILES['csv_file']['tmp_name']);
    $csv_path = $use_uploaded ? $_FILES['csv_file']['tmp_name'] : __DIR__ . DIRECTORY_SEPARATOR . 'CHAMS.csv';

    if (!file_exists($csv_path)) {
        echo json_encode(['status' => 'error', 'message' => 'CSV file not found. Upload one or ensure CHAMS.csv exists in root.']);
        exit;
    }

    $fh = fopen($csv_path, 'r');
    if (!$fh) {
        echo json_encode(['status' => 'error', 'message' => 'Unable to read CSV file.']);
        exit;
    }

    $rows_raw = [];
    while (($row = fgetcsv($fh)) !== false) {
        if ($row === [null] || (count($row) === 1 && trim((string)$row[0]) === '')) { continue; }
        $rows_raw[] = $row;
    }
    fclose($fh);

    if (empty($rows_raw)) {
        echo json_encode(['status' => 'error', 'message' => 'CSV appears empty.']);
        exit;
    }

    // Header detection and column mapping
    $first_lower = array_map(function($v){ return strtolower(trim((string)$v)); }, $rows_raw[0]);
    $has_header = in_array('employee no', $first_lower, true) || in_array('employee_no', $first_lower, true) || in_array('employee code', $first_lower, true) || in_array('chams', $first_lower, true) || in_array('chams_no', $first_lower, true) || in_array('chams number', $first_lower, true) || in_array('chams no', $first_lower, true);
    $col_emp = 0; $col_chams = 1;
    if ($has_header) {
        foreach ($first_lower as $i => $name) {
            if ($name === 'employee no' || $name === 'employee_no' || $name === 'employee code') { $col_emp = $i; }
            if ($name === 'chams' || $name === 'chams_no' || $name === 'chams number' || $name === 'chams no') { $col_chams = $i; }
        }
        array_shift($rows_raw);
    }

    $total = 0; $updatable = 0; $unchanged = 0; $invalid = 0; $missing = 0;
    $detail_rows = [];

    foreach ($rows_raw as $r) {
        $total++;
        $employee_code = trim((string)($r[$col_emp] ?? ''));
        $chams_no = trim((string)($r[$col_chams] ?? ''));

        if ($employee_code === '' || $chams_no === '') {
            $invalid++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'chams_no' => $chams_no,
                'status' => 'invalid',
                'message' => $employee_code === '' ? 'Missing employee_no' : 'Missing chams_no'
            ];
            continue;
        }

        // Employee lookup: try employee_no, fallback to id when numeric
        $stmt = $pdo->prepare("SELECT id, chams_no, full_name, department FROM employees WHERE employee_no = ?");
        $stmt->execute([$employee_code]);
        $emp = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$emp && ctype_digit($employee_code)) {
            $stmt = $pdo->prepare("SELECT id, chams_no, full_name, department FROM employees WHERE id = ?");
            $stmt->execute([intval($employee_code)]);
            $emp = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        if (!$emp) {
            $missing++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'chams_no' => $chams_no,
                'status' => 'missing_employee',
                'message' => 'Employee not found'
            ];
            continue;
        }

        if ((string)$emp['chams_no'] === (string)$chams_no) {
            $unchanged++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'employee_name' => $emp['full_name'],
                'department' => $emp['department'],
                'chams_no' => $chams_no,
                'status' => 'unchanged',
                'message' => 'CHAMS already matches'
            ];
            continue;
        }

        $updatable++;
        $detail_rows[] = [
            'employee_no' => $employee_code,
            'employee_name' => $emp['full_name'],
            'department' => $emp['department'],
            'current_chams_no' => $emp['chams_no'],
            'new_chams_no' => $chams_no,
            'status' => 'ok',
            'message' => 'Will update'
        ];
    }

    echo json_encode([
        'status' => 'ok',
        'summary' => [
            'total_rows' => $total,
            'updatable_rows' => $updatable,
            'unchanged_rows' => $unchanged,
            'invalid_rows' => $invalid,
            'missing_employee_rows' => $missing,
        ],
        'rows' => $detail_rows,
    ]);
    exit;
}

// Upload handler: apply updates to employees.chams_no
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_chams'])) {
    $use_uploaded = isset($_FILES['csv_file']) && is_uploaded_file($_FILES['csv_file']['tmp_name']);
    $csv_path = $use_uploaded ? $_FILES['csv_file']['tmp_name'] : __DIR__ . DIRECTORY_SEPARATOR . 'CHAMS.csv';

    if (!file_exists($csv_path)) {
        $_SESSION['flash'] = 'CSV file not found. Upload one or place CHAMS.csv in root.';
        header('Location: bulk_chams_update.php');
        exit;
    }

    $fh = fopen($csv_path, 'r');
    if (!$fh) {
        $_SESSION['flash'] = 'Unable to read CSV file.';
        header('Location: bulk_chams_update.php');
        exit;
    }
    $rows_raw = [];
    while (($row = fgetcsv($fh)) !== false) {
        if ($row === [null] || (count($row) === 1 && trim((string)$row[0]) === '')) { continue; }
        $rows_raw[] = $row;
    }
    fclose($fh);

    if (empty($rows_raw)) {
        $_SESSION['flash'] = 'CSV appears empty.';
        header('Location: bulk_chams_update.php');
        exit;
    }

    // Header detection and mapping
    $first_lower = array_map(function($v){ return strtolower(trim((string)$v)); }, $rows_raw[0]);
    $has_header = in_array('employee no', $first_lower, true) || in_array('employee_no', $first_lower, true) || in_array('employee code', $first_lower, true) || in_array('chams', $first_lower, true) || in_array('chams_no', $first_lower, true) || in_array('chams number', $first_lower, true) || in_array('chams no', $first_lower, true);
    $col_emp = 0; $col_chams = 1;
    if ($has_header) {
        foreach ($first_lower as $i => $name) {
            if ($name === 'employee no' || $name === 'employee_no' || $name === 'employee code') { $col_emp = $i; }
            if ($name === 'chams' || $name === 'chams_no' || $name === 'chams number' || $name === 'chams no') { $col_chams = $i; }
        }
        array_shift($rows_raw);
    }

    $updated = 0; $unchanged = 0; $invalid = 0; $missing = 0;

    $pdo->beginTransaction();
    try {
        foreach ($rows_raw as $r) {
            $employee_code = trim((string)($r[$col_emp] ?? ''));
            $chams_no = trim((string)($r[$col_chams] ?? ''));

            if ($employee_code === '' || $chams_no === '') { $invalid++; continue; }

            // Robust employee lookup: original employee_no -> digits-only employee_no -> id
            $digits_candidate = null;
            if (preg_match('/^(\d+)/', $employee_code, $m)) { $digits_candidate = $m[1]; }

            $stmt = $pdo->prepare("SELECT id, chams_no FROM employees WHERE employee_no = ?");
            $stmt->execute([$employee_code]);
            $emp = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$emp && $digits_candidate !== null) {
                $stmt = $pdo->prepare("SELECT id, chams_no FROM employees WHERE employee_no = ?");
                $stmt->execute([$digits_candidate]);
                $emp = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            if (!$emp && $digits_candidate !== null) {
                $stmt = $pdo->prepare("SELECT id, chams_no FROM employees WHERE id = ?");
                $stmt->execute([intval($digits_candidate)]);
                $emp = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            if (!$emp) { $missing++; continue; }

            if ((string)$emp['chams_no'] === (string)$chams_no) { $unchanged++; continue; }

            // Update by primary key id to avoid mismatches when CSV uses suffixed codes
            $upd = $pdo->prepare("UPDATE employees SET chams_no = ? WHERE id = ?");
            $upd->execute([$chams_no, $emp['id']]);
            $updated++;
        }
        $pdo->commit();
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['flash'] = 'Error during update: ' . $e->getMessage();
        header('Location: bulk_chams_update.php');
        exit;
    }

    $_SESSION['flash'] = "CHAMS update completed. Updated: $updated, Unchanged: $unchanged, Invalid: $invalid, Missing Employees: $missing.";
    header('Location: bulk_chams_update.php');
    exit;
}
?>
<?php include 'header.php'; ?>

<div class="container py-4">
  <?php if (!empty($_SESSION['flash'])): ?>
    <div class="alert alert-info alert-dismissible fade show" role="alert">
      <?= $_SESSION['flash']; unset($_SESSION['flash']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <h4>Bulk Update CHAMS Number from CSV</h4>
  <p class="text-muted">Upload a CSV with columns like <code>Employee No</code> and <code>chams_no</code>. If no file is uploaded, the system will use <code>CHAMS.csv</code> from the root folder.</p>

  <div class="card mb-3">
    <div class="card-body">
      <form id="chamsForm" method="post" enctype="multipart/form-data">
        <div class="mb-3">
          <label for="csv_file" class="form-label">CSV file (optional)</label>
          <input type="file" class="form-control" name="csv_file" id="csv_file" accept=".csv" />
        </div>
        <div class="d-flex gap-2">
          <button type="button" class="btn btn-primary" onclick="previewChams()">Preview</button>
          <button type="submit" class="btn btn-success" name="upload_chams" value="1">Apply Update</button>
        </div>
        <input type="hidden" name="preview_chams" id="preview_chams" value="0" />
      </form>
    </div>
  </div>

  <div id="previewContainer" class="card" style="display:none;">
    <div class="card-body">
      <h6 class="mb-3">Preview Results</h6>
      <div id="previewSummary" class="mb-2"></div>
      <div class="table-responsive">
        <table class="table table-sm table-bordered">
          <thead>
            <tr>
              <th>Employee No</th>
              <th>Name</th>
              <th>Department</th>
              <th>Current CHAMS</th>
              <th>New CHAMS</th>
              <th>Status</th>
              <th>Message</th>
            </tr>
          </thead>
          <tbody id="previewRows"></tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<script>
function previewChams() {
  const form = document.getElementById('chamsForm');
  const formData = new FormData(form);
  formData.set('preview_chams', '1');

  fetch('bulk_chams_update.php', {
    method: 'POST',
    body: formData
  }).then(r => r.json())
    .then(data => {
      const container = document.getElementById('previewContainer');
      const summaryEl = document.getElementById('previewSummary');
      const rowsEl = document.getElementById('previewRows');
      rowsEl.innerHTML = '';

      if (data.status !== 'ok') {
        container.style.display = 'block';
        summaryEl.innerHTML = `<div class="alert alert-danger">${data.message || 'Preview error'}</div>`;
        return;
      }

      const s = data.summary;
      summaryEl.innerHTML = `
        <div class="alert alert-info">
          Total: ${s.total_rows} | Updatable: ${s.updatable_rows} | Unchanged: ${s.unchanged_rows} | Invalid: ${s.invalid_rows} | Missing Employees: ${s.missing_employee_rows}
        </div>
      `;

      (data.rows || []).forEach(row => {
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td>${row.employee_no || ''}</td>
          <td>${row.employee_name || ''}</td>
          <td>${row.department || ''}</td>
          <td>${row.current_chams_no || ''}</td>
          <td>${row.new_chams_no || row.chams_no || ''}</td>
          <td>${row.status}</td>
          <td>${row.message || ''}</td>
        `;
        rowsEl.appendChild(tr);
      });

      container.style.display = 'block';
    })
    .catch(err => {
      const container = document.getElementById('previewContainer');
      const summaryEl = document.getElementById('previewSummary');
      container.style.display = 'block';
      summaryEl.innerHTML = `<div class="alert alert-danger">${err?.message || err}</div>`;
    });
}
</script>

<?php include 'footer.php'; ?>