<?php
// bulk_cgrade_update.php
// Updates `employees.c_level` and `employees.c_step` using values from CSV (cGrade_level, cGrade_step)

// Make sure session is available for flash messages
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../database/config.php';
// Utility: normalize header names for matching
function norm_header($h) {
    return strtolower(trim(preg_replace('/\s+/', ' ', $h)));
}

// Utility: parse a grade like "14.06", "14/06", "14-06"
function parse_compound_grade($val) {
    if (!is_string($val)) return [null, null];
    $s = trim($val);
    if ($s === '') return [null, null];
    // Extract inner numbers when wrapped like "CONHESS(14.06)"
    if (preg_match('/\((\d{1,2})[\.-\/]?(\d{1,2})\)/', $s, $m)) {
        return [$m[1], ltrim($m[2], '0') === '' ? '0' : ltrim($m[2], '0')];
    }
    if (preg_match('/^(\d{1,2})\s*[\.-\/]\s*(\d{1,2})$/', $s, $m)) {
        return [$m[1], ltrim($m[2], '0') === '' ? '0' : ltrim($m[2], '0')];
    }
    return [null, null];
}

// Utility: robust employee lookup by employee_no or id
function find_employee($pdo, $employeeNoRaw) {
    $employeeNo = trim((string)$employeeNoRaw);
    if ($employeeNo === '') return null;

    // Attempt 1: exact employee_no
    $stmt = $pdo->prepare('SELECT id, employee_no, c_level, c_step FROM employees WHERE employee_no = ? LIMIT 1');
    $stmt->execute([$employeeNo]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) return $row;

    // Attempt 2: digits-only (strip non-digits)
    $digits = preg_replace('/\D+/', '', $employeeNo);
    if ($digits !== '' && $digits !== $employeeNo) {
        $stmt = $pdo->prepare('SELECT id, employee_no, c_level, c_step FROM employees WHERE employee_no = ? LIMIT 1');
        $stmt->execute([$digits]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) return $row;
    }

    // Attempt 3: interpret as id
    if (ctype_digit($digits)) {
        $stmt = $pdo->prepare('SELECT id, employee_no, c_level, c_step FROM employees WHERE id = ? LIMIT 1');
        $stmt->execute([$digits]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) return $row;
    }

    return null;
}

// Reads CSV from uploaded file or a default path
function get_csv_rows(&$headersOut, &$headerIndex) {
    $headersOut = [];
    $headerIndex = [];

    $defaultPath = __DIR__ . DIRECTORY_SEPARATOR . 'staff new data.csv';
    $csvPath = null;

    if (!empty($_FILES['csv_file']['tmp_name'])) {
        $csvPath = $_FILES['csv_file']['tmp_name'];
    } elseif (file_exists($defaultPath)) {
        $csvPath = $defaultPath;
    }

    if (!$csvPath) {
        return [false, 'CSV file not provided and default not found', []];
    }

    $fh = fopen($csvPath, 'r');
    if (!$fh) {
        return [false, 'Unable to open CSV file', []];
    }

    $headers = fgetcsv($fh);
    if (!$headers) {
        fclose($fh);
        return [false, 'CSV is empty or has no header', []];
    }
    $headersOut = $headers;

    // Build header index (normalized)
    foreach ($headers as $i => $h) {
        $headerIndex[norm_header($h)] = $i;
    }

    $rows = [];
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) === 1 && trim((string)$row[0]) === '') {
            continue;
        }
        $rows[] = $row;
    }
    fclose($fh);
    return [true, null, $rows];
}

// Resolve per-row values using flexible header mapping
function resolve_row($headers, $headerIndex, $row) {
    $hmap = function($keyVariants) use ($headerIndex) {
        foreach ($keyVariants as $kv) {
            $nk = norm_header($kv);
            if (isset($headerIndex[$nk])) return $headerIndex[$nk];
        }
        return null;
    };

    // Employee No variants
    $idxEmp = $hmap(['Employee No', 'employee_no', 'EmployeeNo', 'Employee Nos', 'Emp No', 'EmpNo', 'Staff No', 'Staff Number', 'staff_no']);

    // cGrade_level and cGrade_step variants
    $idxCLevel = $hmap(['cGrade_level', 'c_level', 'Current Grade Level', 'C Grade Level', 'cgrade level', 'cgrade_level']);
    $idxCStep  = $hmap(['cGrade_step', 'c_step', 'Current Grade Step', 'C Grade Step', 'cgrade step', 'cgrade_step']);

    $empNo = $idxEmp !== null ? (isset($row[$idxEmp]) ? trim((string)$row[$idxEmp]) : '') : '';
    $cLevel = $idxCLevel !== null ? (isset($row[$idxCLevel]) ? trim((string)$row[$idxCLevel]) : '') : '';
    $cStep  = $idxCStep  !== null ? (isset($row[$idxCStep])  ? trim((string)$row[$idxCStep])  : '') : '';

    // If not found, try to parse compound from known columns like "Pay Grade", "UTH PAY GRADE" or any field with pattern
    if ($cLevel === '' || $cStep === '') {
        $compoundSources = [];
        $compoundCandidates = ['Pay Grade', 'UTH PAY GRADE', 'Grade', 'Current Grade', 'cGrade'];
        foreach ($compoundCandidates as $cand) {
            $idx = $hmap([$cand]);
            if ($idx !== null && isset($row[$idx])) {
                $compoundSources[] = $row[$idx];
            }
        }
        // If nothing from named candidates, scan all cells for a pattern
        if (empty($compoundSources)) {
            foreach ($row as $cell) {
                $compoundSources[] = $cell;
            }
        }
        foreach ($compoundSources as $cs) {
            list($lv, $st) = parse_compound_grade($cs);
            if ($lv !== null && $st !== null) {
                if ($cLevel === '') $cLevel = $lv;
                if ($cStep === '') $cStep = $st;
                break;
            }
        }
    }

    // Also consider explicit columns "grade_level" and "grade_step" as final fallback for cLevel/cStep
    if ($cLevel === '' || $cStep === '') {
        $idxGL = $hmap(['grade_level', 'Grade Level']);
        $idxGS = $hmap(['grade_step', 'Grade Step']);
        if ($idxGL !== null && isset($row[$idxGL])) $cLevel = $cLevel !== '' ? $cLevel : trim((string)$row[$idxGL]);
        if ($idxGS !== null && isset($row[$idxGS])) $cStep  = $cStep  !== '' ? $cStep  : trim((string)$row[$idxGS]);
    }

    // Normalize numeric strings (strip leading zeros from step; keep level as-is)
    $cLevel = preg_match('/^\d{1,2}$/', $cLevel) ? $cLevel : preg_replace('/\D+/', '', $cLevel);
    $cStep  = preg_match('/^\d{1,2}$/', $cStep)  ? ltrim($cStep, '0') : ltrim(preg_replace('/\D+/', '', $cStep), '0');
    if ($cStep === '') $cStep = null; // allow null if cannot parse

    return [$empNo, $cLevel, $cStep];
}

// Handle preview (returns JSON of resolved updates)
if (isset($_POST['preview_cgrade']) && $_POST['preview_cgrade'] === '1') {
    header('Content-Type: application/json');

    $headers = $headerIndex = [];
    list($ok, $err, $rows) = get_csv_rows($headers, $headerIndex);
    if (!$ok) {
        echo json_encode(['ok' => false, 'error' => $err]);
        exit;
    }

    $out = [];
    foreach ($rows as $r) {
        list($empNo, $cLevel, $cStep) = resolve_row($headers, $headerIndex, $r);
        $status = 'ok';
        $note = '';
        $current = null;
        $employee = null;

        if ($empNo === '' || $cLevel === '' || $cStep === null) {
            $status = 'invalid';
            $note = 'Missing employee or cLevel/cStep';
        } else {
            $employee = find_employee($pdo, $empNo);
            if (!$employee) {
                $status = 'missing_employee';
                $note = 'Employee not found';
            } else {
                $current = ['c_level' => $employee['c_level'], 'c_step' => $employee['c_step']];
                if ((string)$employee['c_level'] === (string)$cLevel && (string)$employee['c_step'] === (string)$cStep) {
                    $status = 'unchanged';
                }
            }
        }

        $out[] = [
            'employee_no' => $empNo,
            'resolved' => ['c_level' => $cLevel, 'c_step' => $cStep],
            'current' => $current,
            'status' => $status,
            'note' => $note,
        ];
    }

    echo json_encode(['ok' => true, 'headers' => $headers, 'rows' => $out]);
    exit;
}

// Handle apply update
if (isset($_POST['apply_cgrade']) && $_POST['apply_cgrade'] === '1') {
    $headers = $headerIndex = [];
    list($ok, $err, $rows) = get_csv_rows($headers, $headerIndex);
    if (!$ok) {
        $_SESSION['flash'] = 'Error: ' . $err;
        header('Location: bulk_cgrade_update.php');
        exit;
    }

    $updated = 0; $unchanged = 0; $missing = 0; $invalid = 0; $errors = 0;

    $pdo->beginTransaction();
    try {
        $stmtUpdate = $pdo->prepare('UPDATE employees SET c_level = ?, c_step = ? WHERE id = ?');

        foreach ($rows as $r) {
            list($empNo, $cLevel, $cStep) = resolve_row($headers, $headerIndex, $r);
            if ($empNo === '' || $cLevel === '' || $cStep === null) {
                $invalid++;
                continue;
            }
            $employee = find_employee($pdo, $empNo);
            if (!$employee) {
                $missing++;
                continue;
            }
            if ((string)$employee['c_level'] === (string)$cLevel && (string)$employee['c_step'] === (string)$cStep) {
                $unchanged++;
                continue;
            }

            try {
                $stmtUpdate->execute([$cLevel, $cStep, $employee['id']]);
                $updated++;
            } catch (Exception $e) {
                $errors++;
            }
        }

        $pdo->commit();
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['flash'] = 'Transaction failed: ' . $e->getMessage();
        header('Location: bulk_cgrade_update.php');
        exit;
    }

    $_SESSION['flash'] = "Apply completed. Updated: $updated, Unchanged: $unchanged, Missing: $missing, Invalid: $invalid, Errors: $errors";
    header('Location: bulk_cgrade_update.php');
    exit;
}

// UI (simple form with preview and apply)
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8" />
    <title>Bulk C-Grade Update</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 24px; }
        .container { max-width: 1000px; margin: 0 auto; }
        .flash { background: #e8f4ff; border: 1px solid #b6ddff; padding: 10px; margin-bottom: 16px; }
        .row { border-bottom: 1px solid #eee; padding: 6px 0; }
        .ok { color: #2f6b2f; }
        .invalid { color: #b10000; }
        .missing { color: #b15c00; }
        .unchanged { color: #555; }
        pre { background: #f7f7f7; padding: 8px; overflow: auto; }
    </style>
</head>
<body>
<div class="container">
    <h1>Bulk C-Grade Update (c_level, c_step)</h1>
    <?php if (!empty($_SESSION['flash'])): ?>
        <div class="flash"><?php echo htmlspecialchars($_SESSION['flash']); unset($_SESSION['flash']); ?></div>
    <?php endif; ?>

    <p>Choose a CSV with headers including <code>Employee No</code>, <code>cGrade_level</code>, and <code>cGrade_step</code>. If omitted, the system attempts to parse from <code>Pay Grade</code> (e.g., <code>14.06</code>) or <code>UTH PAY GRADE</code> (e.g., <code>CONHESS(14.06)</code>). Default path: <code>staff new data.csv</code> in the project root.</p>

    <form id="csvForm" method="post" enctype="multipart/form-data" style="margin-bottom: 16px;">
        <label>CSV File: <input type="file" name="csv_file" accept=".csv" /></label>
        <button type="button" onclick="previewCGrade()">Preview</button>
        <button type="submit" onclick="applyCGrade()">Apply Update</button>
        <input type="hidden" name="preview_cgrade" id="preview_cgrade" value="0" />
        <input type="hidden" name="apply_cgrade" id="apply_cgrade" value="0" />
    </form>

    <div id="previewBox"></div>

    <script>
    function previewCGrade() {
        const form = document.getElementById('csvForm');
        document.getElementById('preview_cgrade').value = '1';
        document.getElementById('apply_cgrade').value = '0';

        const fd = new FormData(form);
        fetch('bulk_cgrade_update.php', { method: 'POST', body: fd })
            .then(r => r.json())
            .then(data => {
                const box = document.getElementById('previewBox');
                if (!data.ok) {
                    box.innerHTML = '<div class="invalid">' + (data.error || 'Preview error') + '</div>';
                    return;
                }
                let html = '<h3>Preview</h3>';
                html += '<pre>' + JSON.stringify(data.headers, null, 2) + '</pre>';
                html += '<div>';
                data.rows.slice(0, 200).forEach(r => {
                    const cls = r.status === 'ok' ? 'ok' : (r.status === 'unchanged' ? 'unchanged' : (r.status === 'invalid' ? 'invalid' : 'missing'));
                    html += '<div class="row ' + cls + '">';
                    html += '<strong>' + (r.employee_no || '-') + '</strong> → ' + (r.resolved ? ('c_level=' + (r.resolved.c_level || '-') + ', c_step=' + (r.resolved.c_step || '-')) : '-') + ' ';
                    if (r.current) {
                        html += '(current: c_level=' + (r.current.c_level || '-') + ', c_step=' + (r.current.c_step || '-') + ') ';
                    }
                    html += '[' + r.status + (r.note ? (', ' + r.note) : '') + ']';
                    html += '</div>';
                });
                html += '</div>';
                box.innerHTML = html;
            })
            .catch(err => {
                document.getElementById('previewBox').innerHTML = '<div class="invalid">Network/parse error</div>';
            });
    }
    function applyCGrade() {
        document.getElementById('preview_cgrade').value = '0';
        document.getElementById('apply_cgrade').value = '1';
    }
    </script>
</div>
</body>
</html>