<?php
require_once '../database/config.php';
include 'header.php';

// CSV Allowance Preview Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csv_allowance_preview'])) {
    header('Content-Type: application/json');

    // Form fallbacks (can be overridden by CSV if provided)
    $form_type = $_POST['type'] ?? '';
    $form_date = $_POST['allowance_date'] ?? '';
    $form_recurring = isset($_POST['recurring']) ? 1 : 0;

    if (!$form_type || !$form_date || !isset($_FILES['csv_file'])) {
        echo json_encode(['status' => 'error', 'message' => 'Missing required fields: type, date, and CSV file are required.']);
        exit;
    }

    if (!is_uploaded_file($_FILES['csv_file']['tmp_name'])) {
        echo json_encode(['status' => 'error', 'message' => 'CSV file upload failed.']);
        exit;
    }

    $fh = fopen($_FILES['csv_file']['tmp_name'], 'r');
    if (!$fh) {
        echo json_encode(['status' => 'error', 'message' => 'Unable to read CSV file.']);
        exit;
    }

    $rows_raw = [];
    while (($row = fgetcsv($fh)) !== false) {
        if ($row === [null] || (count($row) === 1 && trim((string)$row[0]) === '')) { continue; }
        $rows_raw[] = $row;
    }
    fclose($fh);

    if (empty($rows_raw)) {
        echo json_encode(['status' => 'error', 'message' => 'CSV file appears to be empty.']);
        exit;
    }

    // Header detection and column mapping
    $first_lower = array_map(function($v){ return strtolower(trim((string)$v)); }, $rows_raw[0]);
    $has_header = in_array('employee no', $first_lower, true) || in_array('employee_no', $first_lower, true) || in_array('employee code', $first_lower, true) || in_array('amount', $first_lower, true) || in_array('recurring', $first_lower, true) || in_array('type', $first_lower, true) || in_array('date', $first_lower, true) || in_array('allowance_date', $first_lower, true);
    $col_emp = 0; $col_amt = 1; $col_rec = 2; $col_type = null; $col_date = null;
    if ($has_header) {
        // Map columns dynamically if header provided
        foreach ($first_lower as $i => $name) {
            if ($name === 'employee no' || $name === 'employee_no' || $name === 'employee code') { $col_emp = $i; }
            if ($name === 'amount') { $col_amt = $i; }
            if ($name === 'recurring' || $name === 'is_recurring') { $col_rec = $i; }
            if ($name === 'type' || $name === 'allowance_type') { $col_type = $i; }
            if ($name === 'date' || $name === 'allowance_date' || $name === 'timestamp') { $col_date = $i; }
        }
        array_shift($rows_raw);
    }

    $total = 0; $valid = 0; $invalid = 0; $duplicates = 0; $missing = 0; $skipped_blank_amount = 0;
    $dept_counts = [];
    $detail_rows = [];

    foreach ($rows_raw as $r) {
        $total++;
        $employee_code = trim((string)($r[$col_emp] ?? ''));
        $amount_raw = trim((string)($r[$col_amt] ?? ''));
        $rec_str = strtolower(trim((string)($r[$col_rec] ?? '')));
        $csv_type = ($col_type !== null) ? trim((string)($r[$col_type] ?? '')) : '';
        $csv_date = ($col_date !== null) ? trim((string)($r[$col_date] ?? '')) : '';

        // Normalize amount: remove commas and currency symbols
        $amount_val = preg_replace('/[^0-9.\-]/', '', $amount_raw);
        if ($amount_val === '') {
            $skipped_blank_amount++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'amount' => $amount_raw,
                'allowance_date' => $csv_date ?: $form_date,
                'recurring' => $form_recurring,
                'status' => 'invalid',
                'message' => 'Blank amount'
            ];
            continue;
        }
        if ($employee_code === '' || !is_numeric($amount_val)) {
            $invalid++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'amount' => $amount_raw,
                'allowance_date' => $csv_date ?: $form_date,
                'recurring' => $form_recurring,
                'status' => 'invalid',
                'message' => $employee_code === '' ? 'Missing employee_no' : 'Amount must be numeric'
            ];
            continue;
        }

        $amount = (float)$amount_val;
        $row_recurring = $form_recurring;
        if ($rec_str !== '') {
            $row_recurring = in_array($rec_str, ['yes','1','true'], true) ? 1 : (in_array($rec_str, ['no','0','false'], true) ? 0 : $form_recurring);
        }
        $row_type = $csv_type ?: $form_type;
        $row_date_raw = $csv_date ?: $form_date;
        // Normalize date: accept timestamp, take YYYY-MM-DD
        $row_date = substr($row_date_raw, 0, 10);

        // Employee lookup: try employee_no first, fallback to id if numeric
        $stmt = $pdo->prepare("SELECT id, full_name, department FROM employees WHERE employee_no = ?");
        $stmt->execute([$employee_code]);
        $emp = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$emp && ctype_digit($employee_code)) {
            $stmt = $pdo->prepare("SELECT id, full_name, department FROM employees WHERE id = ?");
            $stmt->execute([intval($employee_code)]);
            $emp = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        if (!$emp) {
            $missing++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'amount' => $amount,
                'allowance_date' => $row_date,
                'recurring' => $row_recurring,
                'status' => 'missing_employee',
                'message' => 'Employee not found'
            ];
            continue;
        }

        // Duplicate check: same employee, type, amount, date, recurring
        $stmt2 = $pdo->prepare("SELECT COUNT(*) FROM allowances WHERE employee_id = ? AND type = ? AND amount = ? AND allowance_date = ? AND recurring = ?");
        $stmt2->execute([$emp['id'], $row_type, $amount, $row_date, $row_recurring]);
        $is_dup = $stmt2->fetchColumn() > 0;
        if ($is_dup) {
            $duplicates++;
            $detail_rows[] = [
                'employee_no' => $employee_code,
                'employee_name' => $emp['full_name'],
                'department' => $emp['department'],
                'amount' => $amount,
                'allowance_date' => $row_date,
                'recurring' => $row_recurring,
                'status' => 'duplicate',
                'message' => 'Matching allowance already exists'
            ];
            continue;
        }

        $valid++;
        $dept_name = $emp['department'] ?? 'Unknown';
        $dept_counts[$dept_name] = ($dept_counts[$dept_name] ?? 0) + 1;
        $detail_rows[] = [
            'employee_no' => $employee_code,
            'employee_name' => $emp['full_name'],
            'department' => $emp['department'],
            'amount' => $amount,
            'allowance_date' => $row_date,
            'recurring' => $row_recurring,
            'status' => 'ok',
            'message' => 'Ready to insert'
        ];
    }

    echo json_encode([
        'status' => 'ok',
        'summary' => [
            'total_rows' => $total,
            'valid_rows' => $valid,
            'invalid_rows' => $invalid,
            'duplicate_rows' => $duplicates,
            'missing_employee_rows' => $missing,
            'blank_amount_rows' => $skipped_blank_amount,
            'departments' => $dept_counts,
        ],
        'rows' => $detail_rows,
    ]);
    exit;
}


// CSV Allowance Upload Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csv_allowance_upload'])) {
    // Form defaults (can be overridden by CSV if provided)
    $form_type = $_POST['type'] ?? '';
    $form_date = $_POST['allowance_date'] ?? '';
    $form_recurring = isset($_POST['recurring']) ? 1 : 0;

    if (!$form_type || !$form_date || !isset($_FILES['csv_file'])) {
        $_SESSION['flash'] = 'CSV upload failed: type, date, and CSV file are required.';
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='allowances.php';</script>";
        exit;
    }

    if (!is_uploaded_file($_FILES['csv_file']['tmp_name'])) {
        $_SESSION['flash'] = 'CSV upload failed: file not received.';
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='allowances.php';</script>";
        exit;
    }

    $fh = fopen($_FILES['csv_file']['tmp_name'], 'r');
    if (!$fh) {
        $_SESSION['flash'] = 'CSV upload failed: unable to read file.';
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='allowances.php';</script>";
        exit;
    }

    $rows_raw = [];
    while (($row = fgetcsv($fh)) !== false) {
        if ($row === [null] || (count($row) === 1 && trim((string)$row[0]) === '')) { continue; }
        $rows_raw[] = $row;
    }
    fclose($fh);

    if (empty($rows_raw)) {
        $_SESSION['flash'] = 'CSV upload failed: file appears to be empty.';
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='allowances.php';</script>";
        exit;
    }

    // Header detection and dynamic column mapping
    $first_lower = array_map(function($v){ return strtolower(trim((string)$v)); }, $rows_raw[0]);
    $has_header = in_array('employee no', $first_lower, true) || in_array('employee_no', $first_lower, true) || in_array('employee code', $first_lower, true) || in_array('amount', $first_lower, true) || in_array('recurring', $first_lower, true) || in_array('type', $first_lower, true) || in_array('date', $first_lower, true) || in_array('allowance_date', $first_lower, true);
    $col_emp = 0; $col_amt = 1; $col_rec = 2; $col_type = null; $col_date = null;
    if ($has_header) {
        foreach ($first_lower as $i => $name) {
            if ($name === 'employee no' || $name === 'employee_no' || $name === 'employee code') { $col_emp = $i; }
            if ($name === 'amount') { $col_amt = $i; }
            if ($name === 'recurring' || $name === 'is_recurring') { $col_rec = $i; }
            if ($name === 'type' || $name === 'allowance_type') { $col_type = $i; }
            if ($name === 'date' || $name === 'allowance_date' || $name === 'timestamp') { $col_date = $i; }
        }
        array_shift($rows_raw);
    }

    $inserted = 0; $invalid = 0; $duplicates = 0; $missing = 0; $skipped_blank_amount = 0;
    $dept_counts = [];

    $pdo->beginTransaction();
    try {
        foreach ($rows_raw as $r) {
            $employee_code = trim((string)($r[$col_emp] ?? ''));
            $amount_raw = trim((string)($r[$col_amt] ?? ''));
            $rec_str = strtolower(trim((string)($r[$col_rec] ?? '')));
            $csv_type = ($col_type !== null) ? trim((string)($r[$col_type] ?? '')) : '';
            $csv_date = ($col_date !== null) ? trim((string)($r[$col_date] ?? '')) : '';

            // Normalize amount: remove commas and currency symbols
            $amount_val = preg_replace('/[^0-9.\-]/', '', $amount_raw);
            if ($amount_val === '') { $skipped_blank_amount++; continue; }
            if ($employee_code === '' || !is_numeric($amount_val)) { $invalid++; continue; }

            $amount = (float)$amount_val;
            $row_recurring = $form_recurring;
            if ($rec_str !== '') {
                $row_recurring = in_array($rec_str, ['yes','1','true'], true) ? 1 : (in_array($rec_str, ['no','0','false'], true) ? 0 : $form_recurring);
            }
            $row_type = $csv_type ?: $form_type;
            $row_date_raw = $csv_date ?: $form_date;
            // Normalize date: accept timestamp, take YYYY-MM-DD
            $row_date = substr($row_date_raw, 0, 10);

            // Employee lookup: try employee_no first, fallback to id if numeric
            $stmt = $pdo->prepare("SELECT id, department FROM employees WHERE employee_no = ?");
            $stmt->execute([$employee_code]);
            $emp = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$emp && ctype_digit($employee_code)) {
                $stmt = $pdo->prepare("SELECT id, department FROM employees WHERE id = ?");
                $stmt->execute([intval($employee_code)]);
                $emp = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            if (!$emp) { $missing++; continue; }

            // Duplicate check: same employee, type, amount, date, recurring
            $stmt2 = $pdo->prepare("SELECT COUNT(*) FROM allowances WHERE employee_id = ? AND type = ? AND amount = ? AND allowance_date = ? AND recurring = ?");
            $stmt2->execute([$emp['id'], $row_type, $amount, $row_date, $row_recurring]);
            $is_dup = $stmt2->fetchColumn() > 0;
            if ($is_dup) { $duplicates++; continue; }

            // Insert allowance
            $ins = $pdo->prepare("INSERT INTO allowances (employee_id, type, amount, allowance_date, recurring) VALUES (?, ?, ?, ?, ?)");
            $ins->execute([$emp['id'], $row_type, $amount, $row_date, $row_recurring]);
            $inserted++;

            // Department summary
            $dept_name = $emp['department'] ?? 'Unknown';
            $dept_counts[$dept_name] = ($dept_counts[$dept_name] ?? 0) + 1;
        }
        $pdo->commit();
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['flash'] = 'CSV upload error: ' . $e->getMessage();
        log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
        echo "<script>window.location.href='allowances.php';</script>";
        exit;
    }

    $dept_summary = [];
    foreach ($dept_counts as $d => $c) { $dept_summary[] = "$d: $c"; }
    $dept_summary_str = empty($dept_summary) ? 'None' : implode(', ', $dept_summary);

    $_SESSION['flash'] = "CSV allowances uploaded. Inserted: $inserted, Duplicates: $duplicates, Invalid: $invalid, Missing Employees: $missing, Blank Amounts: $skipped_blank_amount. Departments: $dept_summary_str.";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
    exit;
}

// Handle Add/Edit Allowance
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['employee_id'])) {
    $employee_id = intval($_POST['employee_id']);
    $type = $_POST['type'];
    $amount = floatval($_POST['amount']);
    $allowance_date = $_POST['allowance_date'];
    $recurring = isset($_POST['recurring']) ? 1 : 0;
    $edit_id = isset($_POST['edit_id']) ? intval($_POST['edit_id']) : 0;

    if ($edit_id > 0) {
        $stmt = $pdo->prepare("UPDATE allowances SET employee_id=?, type=?, amount=?, allowance_date=?, recurring=? WHERE id=?");
        $stmt->execute([$employee_id, $type, $amount, $allowance_date, $recurring, $edit_id]);
    } else {
        $stmt = $pdo->prepare("INSERT INTO allowances (employee_id, type, amount, allowance_date, recurring) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$employee_id, $type, $amount, $allowance_date, $recurring]);
    }

    $_SESSION['flash'] = "$type Allowance saved successfully.";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
}

// Handle Bulk Allowance
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_allowance'])) {
    $department = $_POST['department'];
    $type = $_POST['type'];
    $amount = floatval($_POST['amount']);
    $allowance_date = $_POST['allowance_date'];
    $recurring = isset($_POST['recurring']) ? 1 : 0;

    $stmt = $pdo->prepare("SELECT id FROM employees WHERE department = ?");
    $stmt->execute([$department]);
    $employees = $stmt->fetchAll(PDO::FETCH_COLUMN);

    foreach ($employees as $emp_id) {
        try {
            $stmt = $pdo->prepare("INSERT INTO allowances (employee_id, type, amount, allowance_date, recurring) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$emp_id, $type, $amount, $allowance_date, $recurring]);
        } catch (PDOException $e) {}
    }

    $_SESSION['flash'] = "$type Bulk allowance applied to $department department.";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
}

// Handle Bulk by Grade Level/Scheme/Employee Type
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_grade_scheme'])) {
    $scheme = $_POST['scheme'];
    $employee_type = $_POST['employee_type'];
    $grade_level = $_POST['grade_level'];
    $step = $_POST['step'];
    $type = $_POST['type'];
    $amount = floatval($_POST['amount']);
    $allowance_date = $_POST['allowance_date'];
    $recurring = isset($_POST['recurring']) ? 1 : 0;

    $stmt = $pdo->prepare("SELECT id FROM employees WHERE scheme = ? AND employee_type = ? AND grade_level = ? AND step = ?");
    $stmt->execute([$scheme, $employee_type, $grade_level, $step]);
    $employees = $stmt->fetchAll(PDO::FETCH_COLUMN);

    foreach ($employees as $emp_id) {
        try {
            $stmt2 = $pdo->prepare("INSERT INTO allowances (employee_id, type, amount, allowance_date, recurring) VALUES (?, ?, ?, ?, ?)");
            $stmt2->execute([$emp_id, $type, $amount, $allowance_date, $recurring]);
        } catch (PDOException $e) {}
    }

    $_SESSION['flash'] = "$type Bulk allowance applied to Grade $grade_level - $step ($scheme / $employee_type).";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
    echo "<noscript><meta http-equiv='refresh' content='0;url=allowances.php'></noscript>";
    exit;
}


// Handle Add/Delete Type
if (isset($_POST['add_type'])) {
    $new_type = trim($_POST['new_type']);
    if (!empty($new_type)) {
        try {
            $stmt = $pdo->prepare("INSERT INTO allowance_types (name) VALUES (?)");
            $stmt->execute([$new_type]);
            $_SESSION['flash'] = "New allowance type added. $new_type";
        } catch (PDOException $e) {
            $_SESSION['flash'] = "Type may already exist.";
        }
    }
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
}

if (isset($_GET['delete_type'])) {
    $id = intval($_GET['delete_type']);
    $pdo->prepare("DELETE FROM allowance_types WHERE id = ?")->execute([$id]);
    $_SESSION['flash'] = "Allowance type deleted.";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
}

// Handle Delete
if (isset($_GET['delete_id'])) {
    $id = intval($_GET['delete_id']);
    $pdo->prepare("DELETE FROM allowances WHERE id = ?")->execute([$id]);
    $_SESSION['flash'] = "Allowance deleted.";
    log_audit($pdo, $_SESSION['user_id'], $_SESSION['flash']);
    echo "<script>window.location.href='allowances.php';</script>";
}

$employees = $pdo->query("SELECT id, full_name, employee_no, department FROM employees ORDER BY full_name")->fetchAll(PDO::FETCH_ASSOC);
$departments = $pdo->query("SELECT DISTINCT department FROM employees WHERE department IS NOT NULL ORDER BY department")->fetchAll(PDO::FETCH_COLUMN);
$types = $pdo->query("SELECT * FROM allowance_types ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

// FILTER BY EMPLOYEE CODE
$filter_emp_code = $_GET['employee_code'] ?? '';
if ($filter_emp_code) {
    $stmt = $pdo->prepare("SELECT a.*, e.full_name, e.employee_no AS employee_code FROM allowances a JOIN employees e ON a.employee_id = e.id WHERE e.employee_no = ?");
    $stmt->execute([$filter_emp_code]);
} else {
    $stmt = $pdo->query("SELECT a.*, e.full_name, e.employee_no AS employee_code FROM allowances a JOIN employees e ON a.employee_id = e.id");
}
$allowances = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<?php  
?>
<div class="container py-4">
    <?php if (!empty($_SESSION['flash'])): ?>
        <div class="alert alert-info alert-dismissible fade show" role="alert">
            <?= $_SESSION['flash']; unset($_SESSION['flash']); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

<!-- Button to trigger modal -->
<div class="d-flex justify-content-end mb-3">
  <button class="btn btn-outline-primary me-2" data-bs-toggle="modal" data-bs-target="#bulkAllowanceModal">Bulk Allowances</button>
  <button class="btn btn-outline-success me-2" data-bs-toggle="modal" data-bs-target="#csvAllowanceModal">CSV Upload</button>
  <a class="btn btn-outline-dark" href="bulk_grade_step_upload.php">Grade/Step CSV Update</a>
</div>

    <div class="d-flex justify-content-between align-items-center mb-3">
        <h4>Allowances</h4>
        <div>
            <button class="btn btn-secondary" data-bs-toggle="modal" data-bs-target="#typeModal">Manage Types</button>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#allowanceModal" onclick="openAddModal()">Add Allowance</button>
        </div>
    </div>

    <!-- Filter by Employee Code -->
    <form class="row g-3 mb-3" method="get">
        <div class="col-md-4">
            <label for="employee_code" class="form-label">Filter by Employee Code</label>
            <input type="text" name="employee_code" id="employee_code_filter" class="form-control"
                   value="<?= isset($_GET['employee_code']) ? htmlspecialchars($_GET['employee_code']) : '' ?>"
                   placeholder="Enter Employee Code (e.g., EMP001)">
        </div>
        <div class="col-md-2 align-self-end">
            <button class="btn btn-primary btn-sm" type="submit">Filter</button>
        </div>
    </form>


    <table id="allowanceTable" class="table table-bordered table-striped">
        <thead>
            <tr>
                <th>ID</th>
                <th>Employee</th>
                <th>Type</th>
                <th>Amount (₦)</th>
                <th>Date</th>
                <th>Recurring</th>
                <th>Description</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($allowances as $row): ?>
                <tr>
                    <td><?= $row['id'] ?></td>
                    <td><?= htmlspecialchars($row['full_name']) ?></td>
                    <td><?= $row['type'] ?></td>
                    <td><?= number_format($row['amount'], 2) ?></td>
                    <td><?= $row['allowance_date'] ?></td>
                    <td><?= $row['recurring'] ? 'Yes' : 'No' ?></td>
                    <td><?= htmlspecialchars($row['description']) ?></td>
                    <td>
                        <button class="btn btn-sm btn-warning" onclick='openEditModal(<?= json_encode($row) ?>)'>Edit</button>
                        <a href="allowances.php?delete_id=<?= $row['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this allowance?')">Delete</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>
<!-- Add/Edit Allowance Modal -->
<div class="modal fade" id="allowanceModal" tabindex="-1" aria-labelledby="allowanceModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="allowanceModalLabel">Add/Edit Allowance</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="edit_id" id="edit_id">
        <div class="mb-3">
          <label>Employee Code</label>
          <input type="text" name="employee_code" id="employee_code" class="form-control" onblur="fetchEmployeeName(this.value)" required>
          <input type="hidden" name="employee_id" id="employee_id_input">
        </div>
        <div class="mb-3">
          <label>Employee Name</label>
          <input type="text" id="employee_name_display" class="form-control" readonly>
        </div>
        <div class="mb-3">
          <label>Type</label>
          <select name="type" id="type" class="form-select" required>
            <option value="">-- Select Type --</option>
            <?php foreach ($types as $t): ?>
              <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-3">
          <label>Amount (₦)</label>
          <input type="number" step="0.01" name="amount" id="amount" class="form-control" required>
        </div>
        <div class="mb-3">
          <label>Date</label>
          <input type="date" name="allowance_date" id="allowance_date" class="form-control" required>
        </div>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" name="recurring" id="recurring">
          <label class="form-check-label">Recurring</label>
        </div>
      </div>
      <div class="modal-footer">
        <button type="submit" class="btn btn-success">Save</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<!-- Bulk Allowance Modal (Tabs) -->
<div class="modal fade" id="bulkAllowanceModal" tabindex="-1" aria-labelledby="bulkAllowanceModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="bulkAllowanceModalLabel">Bulk Allowances</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">

        <!-- Nav tabs -->
        <ul class="nav nav-tabs mb-3" id="bulkTabs" role="tablist">
          <li class="nav-item" role="presentation">
            <button class="nav-link active" id="dept-tab" data-bs-toggle="tab" data-bs-target="#deptTab" type="button" role="tab">By Department</button>
          </li>
          <li class="nav-item" role="presentation">
            <button class="nav-link" id="grade-tab" data-bs-toggle="tab" data-bs-target="#gradeTab" type="button" role="tab">By Grade/Scheme/Type</button>
          </li>
        </ul>

        <div class="tab-content">
          <!-- By Department -->
          <div class="tab-pane fade show active" id="deptTab" role="tabpanel">
            <form method="post">
              <input type="hidden" name="bulk_allowance" value="1">
              <div class="mb-3">
                <label class="form-label">Department</label>
                <select name="department" class="form-select" required>
                  <option value="">-- Select Department --</option>
                  <?php foreach ($departments as $dept): ?>
                    <option value="<?= htmlspecialchars($dept) ?>"><?= htmlspecialchars($dept) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Type</label>
                <select name="type" class="form-select" required>
                  <option value="">-- Select Type --</option>
                  <?php foreach ($types as $t): ?>
                    <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Amount</label>
                <input type="number" step="0.01" name="amount" class="form-control" required>
              </div>
              <div class="mb-3">
                <label class="form-label">Date</label>
                <input type="date" name="allowance_date" class="form-control" required>
              </div>
              <div class="form-check mb-3">
                <input class="form-check-input" type="checkbox" name="recurring" id="dept_recurring">
                <label class="form-check-label" for="dept_recurring">Recurring</label>
              </div>
              <button type="submit" class="btn btn-success">Apply (Department)</button>
            </form>
          </div>
          <!-- By Grade/Scheme/Employee Type -->
          <div class="tab-pane fade" id="gradeTab" role="tabpanel">
            <form method="post" id="bulkGradeForm">
              <input type="hidden" name="bulk_grade_scheme" value="1">
              <div class="mb-3">
                <label class="form-label">Scheme</label>
                <select name="scheme" id="bulk_scheme" class="form-select" required>
                  <option value="">-- Select Scheme --</option>
                  <?php 
                  $schemes = $pdo->query("SELECT DISTINCT scheme FROM employees WHERE scheme IS NOT NULL AND scheme <> '' ORDER BY scheme")->fetchAll(PDO::FETCH_COLUMN);
                  foreach ($schemes as $sch): ?>
                    <option value="<?= htmlspecialchars($sch) ?>"><?= htmlspecialchars($sch) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Employee Type</label>
                <select name="employee_type" id="bulk_employee_type" class="form-select" required>
                  <option value="">-- Select Employee Type --</option>
                  <!-- Fills by JS -->
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Grade Level</label>
                <select name="grade_level" id="bulk_grade_level" class="form-select" required>
                  <option value="">-- Select Grade Level --</option>
                  <!-- Fills by JS -->
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Step</label>
                <select name="step" id="step" class="form-select" required>
                  <option disabled value="">-- Select Step --</option>
                  <option>1</option>
                  <option>2</option>
                  <option>3</option>
                  <option>4</option>
                  <option>5</option>
                  <option>6</option>
                  <option>7</option>
                  <option>8</option>
                  <option>9</option>
                  <option>10</option>
                  <option>11</option>
                  <option>12</option>
                  <option>13</option>
                  <option>14</option>
                  <option>15</option>
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Type</label>
                <select name="type" class="form-select" required>
                  <option value="">-- Select Type --</option>
                  <?php foreach ($types as $t): ?>
                    <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="mb-3">
                <label class="form-label">Amount</label>
                <input type="number" step="0.01" name="amount" class="form-control" required>
              </div>
              <div class="mb-3">
                <label class="form-label">Date</label>
                <input type="date" name="allowance_date" class="form-control" required>
              </div>
              <div class="form-check mb-3">
                <input class="form-check-input" type="checkbox" name="recurring" id="bulk_grade_recurring">
                <label class="form-check-label" for="bulk_grade_recurring">Recurring</label>
              </div>
              <button type="submit" class="btn btn-success">Apply (Grade/Scheme/Type)</button>
            </form>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- CSV Allowance Upload Modal -->
<div class="modal fade" id="csvAllowanceModal" tabindex="-1" aria-labelledby="csvAllowanceModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" enctype="multipart/form-data" class="modal-content" id="csvAllowanceForm">
      <div class="modal-header">
        <h5 class="modal-title" id="csvAllowanceModalLabel">CSV Allowance Upload</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="csv_allowance_upload" value="1">
        <div class="mb-3">
          <label class="form-label">Type</label>
          <select name="type" class="form-select" required>
            <option value="">-- Select Type --</option>
            <?php foreach ($types as $t): ?>
              <option value="<?= htmlspecialchars($t['name']) ?>"><?= htmlspecialchars($t['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-3">
          <label class="form-label">Date</label>
          <input type="date" name="allowance_date" class="form-control" required>
        </div>
        <div class="form-check mb-3">
          <input class="form-check-input" type="checkbox" name="recurring" id="csv_allowance_recurring">
          <label class="form-check-label" for="csv_allowance_recurring">Recurring</label>
        </div>
        <div class="mb-3">
          <label class="form-label">CSV File</label>
          <input type="file" name="csv_file" class="form-control" accept=".csv,text/csv" required>
          <small class="form-text text-muted">
            CSV columns: employee_no, amount[, recurring]. Department auto-detected.
          </small>
        </div>
        <div id="csvAllowancePreviewContainer" class="mt-3"></div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-info" onclick="previewCsvAllowance()">Preview</button>
        <button type="submit" class="btn btn-success">Upload</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </form>
  </div>
</div>

<!-- Manage Allowance Types Modal -->
<div class="modal fade" id="typeModal" tabindex="-1" aria-labelledby="typeModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="typeModalLabel">Manage Allowance Types</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <label for="new_type" class="form-label">New Type</label>
          <input type="text" name="new_type" id="new_type" class="form-control" required>
          <input type="hidden" name="add_type" value="1">
        </div>
        <hr>
        <ul class="list-group">
          <?php foreach ($types as $t): ?>
            <li class="list-group-item d-flex justify-content-between align-items-center">
              <?= htmlspecialchars($t['name']) ?>
              <a href="?delete_type=<?= $t['id'] ?>" onclick="return confirm('Delete this type?')" class="btn btn-sm btn-danger">Delete</a>
            </li>
          <?php endforeach; ?>
        </ul>
      </div>
      <div class="modal-footer">
        <button type="submit" class="btn btn-success">Add</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </form>
  </div>
</div>


<script>
$(document).ready(function () {
    $('#allowanceTable').DataTable();
});

$(function() {
    $('#bulk_scheme').on('change', function() {
        var scheme = $(this).val();
        $('#bulk_employee_type').html('<option value="">Loading...</option>');
        $('#bulk_grade_level').html('<option value="">-- Select Grade Level --</option>');
        if (scheme) {
            $.post('ajax_allowance_helpers.php', {action: 'employee_types_for_scheme', scheme: scheme}, function(data) {
                var opts = '<option value="">-- Select Employee Type --</option>';
                $.each(data, function(i, v) { opts += '<option value="'+v+'">'+v+'</option>'; });
                $('#bulk_employee_type').html(opts);
            }, 'json');
        }
    });

    $('#bulk_employee_type').on('change', function() {
        var scheme = $('#bulk_scheme').val();
        var empType = $(this).val();
        $('#bulk_grade_level').html('<option value="">Loading...</option>');
        if (scheme && empType) {
            $.post('ajax_allowance_helpers.php', {action: 'grades_for_type_scheme', scheme: scheme, employee_type: empType}, function(data) {
                var opts = '<option value="">-- Select Grade Level --</option>';
                $.each(data, function(i, v) { opts += '<option value="'+v+'">'+v+'</option>'; });
                $('#bulk_grade_level').html(opts);
            }, 'json');
        }
    });
});


function fetchEmployeeName(code) {
    if (code.trim() === "") {
        $('#employee_name_display').val('');
        $('#employee_id_input').val('');
        return;
    }

    $.post('fetch_employee_name.php', { code: code }, function(res) {
        if (res.status === 'success') {
            $('#employee_name_display').val(res.name);
            $('#employee_id_input').val(res.id);
        } else {
            $('#employee_name_display').val('Not found');
            $('#employee_id_input').val('');
        }
    }, 'json');
}

function openAddModal() {
    $('#allowanceModalLabel').text("Add Allowance");
    $('#edit_id').val('');
    $('#employee_code').val('');
    $('#employee_id_input').val('');
    $('#employee_name_display').val('');
    $('#type').val('');
    $('#amount').val('');
    $('#allowance_date').val('');
    $('#recurring').prop('checked', false);
}

function openEditModal(data) {
    $('#allowanceModalLabel').text("Edit Allowance");
    $('#edit_id').val(data.id);
    $('#employee_code').val(data.employee_code || '');
    $('#type').val(data.type);
    $('#amount').val(data.amount);
    $('#allowance_date').val(data.allowance_date);
    $('#recurring').prop('checked', data.recurring == 1);

    if (data.employee_code) {
        fetchEmployeeName(data.employee_code);
    } else {
        $('#employee_name_display').val('Not found');
        $('#employee_id_input').val('');
    }

    new bootstrap.Modal(document.getElementById('allowanceModal')).show();
}
</script>

<script>
function previewCsvAllowance() {
  var form = document.getElementById('csvAllowanceForm');
  if (!form) return;
  var fd = new FormData(form);
  fd.append('csv_allowance_preview', '1');

  var container = document.getElementById('csvAllowancePreviewContainer');
  if (container) {
    container.innerHTML = '<div class="alert alert-info">Generating preview...</div>';
  }

  $.ajax({
    url: 'allowances.php',
    method: 'POST',
    data: fd,
    processData: false,
    contentType: false,
    dataType: 'json',
    success: function(res) {
      if (!container) return;
      if (res.status !== 'ok') {
        container.innerHTML = '<div class="alert alert-danger">'+(res.message || 'Preview failed')+'</div>';
        return;
      }
      var s = res.summary || {};
      var deptSummary = s.departments || {};
      var deptList = Object.keys(deptSummary).map(function(d){ return d+': '+deptSummary[d]; }).join(', ');
      var html = '';
      html += '<div class="alert alert-success">';
      html += 'Total: '+(s.total_rows||0)+', Valid: '+(s.valid_rows||0)+', Duplicates: '+(s.duplicate_rows||0)+', Missing: '+(s.missing_employee_rows||0)+', Invalid: '+(s.invalid_rows||0);
      html += '</div>';
      html += '<div><strong>Departments:</strong> '+(deptList || 'None')+'</div>';

      html += '<div class="table-responsive mt-2"><table class="table table-sm table-bordered"><thead><tr><th>Employee No</th><th>Name</th><th>Department</th><th>Amount</th><th>Date</th><th>Recurring</th><th>Status</th><th>Message</th></tr></thead><tbody>';
      (res.rows || []).forEach(function(r){
        html += '<tr>';
        html += '<td>'+(r.employee_no||'')+'</td>';
        html += '<td>'+(r.employee_name||'')+'</td>';
        html += '<td>'+(r.department||'')+'</td>';
        html += '<td>'+(r.amount||'')+'</td>';
        html += '<td>'+(r.allowance_date||'')+'</td>';
        html += '<td>'+(r.recurring==1?'Yes':'No')+'</td>';
        var badgeClass = (r.status==='ok') ? 'bg-success' : (r.status==='duplicate' ? 'bg-warning' : (r.status==='missing_employee' ? 'bg-danger' : 'bg-secondary'));
        html += '<td><span class="badge '+badgeClass+'">'+(r.status||'')+'</span></td>';
        html += '<td>'+(r.message||'')+'</td>';
        html += '</tr>';
      });
      html += '</tbody></table></div>';
      container.innerHTML = html;
    },
    error: function(xhr) {
      if (container) {
        container.innerHTML = '<div class="alert alert-danger">Preview request failed.</div>';
      }
    }
  });
}
</script>

<?php include 'footer.php'?>